#ifndef DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_REPOSITORY_H_
#define DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_REPOSITORY_H_

#include <macs_pg/subscription/subscription.h>
#include <src/access/subscription_repository.h>
#include <src/access_impl/timer.h>
#include <src/detail/guarded.h>
#include <src/access_impl/wrap_yield.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/numeric/conversion/cast.hpp>
#include <chrono>

BOOST_FUSION_DEFINE_STRUCT((doberman)(access_impl), SubscriptionRepositoryTimes,
        (std::uint64_t, sleep_seconds)
        (std::uint64_t, least_wait_ms)
)


namespace doberman {
namespace access_impl {

template <typename IO, typename Clock = std::chrono::steady_clock>
class SubscriptionRepository {
    using Duration = typename Clock::duration;
    using Timepoint = typename Clock::time_point;
    IO io_;
    const SubscriptionRepositoryTimes times_;

    decltype(auto) io() const {
        return ::doberman::detail::dereference(io_);
    }

    Seconds sleepTime() const {
        return Seconds(times_.sleep_seconds);
    }

public:
    using Data = std::vector<::doberman::logic::SubscriptionData>;

    SubscriptionRepository(IO io, SubscriptionRepositoryTimes times)
    : io_(std::move(io)), times_(std::move(times)) {}

    auto makeContext() const {
        return std::make_tuple(Clock::now() - sleepTime());
    }

    template <typename Ctx, typename Yield>
    Data reserve(Ctx& ctx, int credit, Yield yield) const {
        holdTimeout(ctx, yield);
        return credit ? makeData(io().reserve(
                boost::numeric_cast<std::size_t>(credit), yield)) : Data{};
    }

    template <typename Ctx, typename Yield>
    Data getReserved(const Ctx&, Yield yield) const {
        return makeData(io().getReserved(yield));
    }

    template <typename Ctx, typename Yield>
    void release(const Ctx&, SubscriptionId sid, Yield yield) const {
        io().release(sid.uid, sid.id, yield);
    }

    template <typename Ctx, typename Yield>
    void decline(const Ctx&, SubscriptionId, std::string, Yield) const {
    }

private:
    template <typename Range>
    Data makeData(Range&& range) const {
        auto retval = range | boost::adaptors::transformed([](auto& s){
            return ::doberman::logic::SubscriptionData {
                {s.uid(), s.subscriptionId()},
                {{s.uid()}, s.fid()},
                {s.subscriberUid()}
            };
        });
        return {retval.begin(), retval.end()};
    }

    template <typename Ctx>
    static auto& lastWaitTime(Ctx& ctx) {
        return std::get<0>(ctx);
    }

    static auto now() { return Clock::now(); }

    template <typename Ctx, typename Yield>
    void holdTimeout(Ctx& ctx, Yield yield) const {
        auto period = now() - lastWaitTime(ctx);
        if (period < sleepTime()) {
            timer::wait(sleepTime() - period, yield);
        }
        lastWaitTime(ctx) = now();
    }
};

template <typename IO>
inline auto makeSubscriptionRepository(IO io, SubscriptionRepositoryTimes times) {
    return SubscriptionRepository<IO>(std::move(io), std::move(times));
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_REPOSITORY_H_ */
