#pragma once

#include <src/detail/magic_hat.h>
#include <src/logic/types.h>
#include <yamail/resource_pool/async/pool.hpp>
#include <coroutine_mutex/coroutine_mutex.hpp>
#include <boost/asio/io_service.hpp>
#include <boost/asio/spawn.hpp>
#include <vector>

namespace doberman {
namespace access_impl {

template <typename Mutex, typename GetResource>
class SubscriptionResource {
    struct MutexCreator {
        std::shared_ptr<Mutex> operator() (Uid) const {
            return std::make_shared<Mutex>();
        }
    };

    GetResource getResource_;
    detail::MagicHat<Uid, MutexCreator> hat_;
public:

    SubscriptionResource(GetResource f) : getResource_{std::move(f)}, hat_({}){}

    template <typename Yield>
    auto get(const logic::Subscriber& s, Yield yield) {
        using UidMutexRef = decltype(hat_.get(s.uid));
        using PoolHandle = decltype(getResource_(yield));
        struct Handle {
            UidMutexRef uidMutex_;
            coro::unique_lock<Mutex> uidLock_;
            PoolHandle initHandle_;

            Handle(UidMutexRef uidMutex, GetResource& getResource, Yield yield)
            : uidMutex_(uidMutex),
              uidLock_(*uidMutex_, yield),
              initHandle_(getResource(yield))
            {}
        };
        return Handle{hat_.get(s.uid), getResource_, yield};
    }
};

inline auto makeSubscriptionResource(boost::asio::io_service& ios,
        std::size_t maxSubscriptionHeavyProcessCount,
        std::size_t maxSubscriptionsCount) {
    if (maxSubscriptionHeavyProcessCount == 0) {
        throw std::invalid_argument("init resource pool size can not be 0");
    }
    using Pool = yamail::resource_pool::async::pool<int>;
    auto getResource = [p = Pool{[]{ return 1;}, maxSubscriptionHeavyProcessCount, maxSubscriptionsCount}, &ios]
                        (boost::asio::yield_context yield) mutable {
        return p.get_auto_recycle(ios, yield, yamail::resource_pool::time_traits::duration::max());
    };
    return SubscriptionResource<coro::Mutex, decltype(getResource)>{std::move(getResource)};
}

} // namespace access_impl
} // namespace doberman
