#pragma once

#include <coroutine_mutex/coroutine_mutex.hpp>
#include <boost/range/algorithm.hpp>

namespace doberman {
namespace access_impl {

/**
 * Lightweight non thread-safe condvar implementation
 * - need to be removed for boost::fibers::condition_variable
 */
class WaitList {
public:
    using YieldContext = boost::asio::yield_context;

    WaitList() = default;
    WaitList(const WaitList&) = delete;
    WaitList& operator = (const WaitList&) = delete;

    void wait(const YieldContext& yield) {
        enqueue(yield);
        suspend(yield);
    }

    void notify_all() {
        std::vector<Consumer> tmp;
        std::swap(tmp, queue_);
        boost::for_each(tmp, [&](auto& c) {awake(c);});
    }

    ~WaitList() {
        notify_all();
    }
private:
    struct Consumer {
        boost::asio::detail::shared_ptr<YieldContext::callee_type> coro;
    #ifdef CORO_MUTEX_USE_OLD_ASIO_INTERFACE
        boost::asio::strand dispatcher;
    #else
        boost::asio::executor executor;
    #endif
    };

    static_assert(std::is_nothrow_move_constructible<Consumer>::value, "Consumer must be nothrow move constructible");

    void enqueue(YieldContext yield) {
    #ifdef CORO_MUTEX_USE_OLD_ASIO_INTERFACE
        queue_.push_back(Consumer {yield.coro_.lock(), yield.handler_.dispatcher_});
    #else
        queue_.push_back(Consumer {yield.coro_.lock(), boost::asio::get_associated_executor(yield.handler_)});
    #endif
    }

    static void suspend(YieldContext yield) {
        yield.ca_();
    }

    static void awake(Consumer consumer) {
    #ifdef CORO_MUTEX_USE_OLD_ASIO_INTERFACE
        consumer.dispatcher.post([coro = std::move(consumer.coro)] { (*coro)(); });
    #else
        boost::asio::post(consumer.executor, [coro = std::move(consumer.coro)] { (*coro)(); });
    #endif
    }

    std::vector<Consumer> queue_;
};

} // namespace access_impl
} // namespace doberman
