#ifndef DOBERMAN_SRC_LOGGER_SPDLOG_H_
#define DOBERMAN_SRC_LOGGER_SPDLOG_H_


#include <logdog/format/tskv.h>
#include <logdog/backend/spdlog.h>
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wold-style-cast"
#include <spdlog/spdlog.h>
#pragma GCC diagnostic pop
#include <boost/fusion/adapted.hpp>

BOOST_FUSION_DEFINE_STRUCT((doberman)(log)(spd), SpdlogConfig,
        (std::int64_t, flush_interval_ms)
        (bool, force_flush)
        (std::size_t, queue_size)
        (std::string, filename)
        (std::string, level)
)

namespace doberman {
namespace log {
namespace spd {

class Sink : public ::spdlog::sinks::sink {
public:
    Sink(const std::string& filename, bool forceFlush = false)
            : file_helper_(forceFlush) {
        file_helper_.open(filename);
    }

    void reopen() {
        needReopen_ = true;
    }

private:
    spdlog::details::file_helper file_helper_;
    std::atomic<bool> needReopen_{false};

    void flush() override final {
        file_helper_.flush();
    }

    void log(const spdlog::details::log_msg& msg) override final {
        if( needReopen_ ) {
            flush();
            file_helper_.reopen(false);
            needReopen_ = false;
        }
        file_helper_.write(msg);
    }
};

inline auto makeSink(SpdlogConfig cfg) {
    return std::make_shared<Sink>(cfg.filename, cfg.force_flush);
}

inline auto makeSpdLog(SpdlogConfig cfg, const std::string& name, std::shared_ptr<Sink> sink) {
    auto spd = std::make_shared<spdlog::async_logger>(
                name,
                sink,
                cfg.queue_size,
                spdlog::async_overflow_policy::discard_log_msg,
                nullptr,
                cfg.flush_interval_ms != 0 ?
                        std::chrono::milliseconds(cfg.flush_interval_ms) :
                        std::chrono::milliseconds::zero(),
                nullptr
            );
    using ::spdlog::level::level_names;
    using ::spdlog::level::level_enum;
    const auto i = std::find(std::begin(level_names), std::end(level_names), cfg.level);
    if (i==std::end(level_names)) {
        throw std::invalid_argument("bad log level name" + cfg.level);
    }
    spd->set_level(level_enum(std::distance(std::begin(level_names), i)));
    return spd;
}

constexpr static auto dobby_formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-doberman-log"));

inline auto make_log(spd::SpdlogConfig cfg) {
    auto sink = spd::makeSink(cfg);
    auto spd = spd::makeSpdLog(cfg, "tskv", sink);
    spd->set_formatter(std::make_shared<spdlog::pattern_formatter>("%v"));
    auto log = ::logdog::make_log(dobby_formatter, spd);
    return std::make_tuple(log, sink);
}

} // namespace spd
} // namespace log
} // namespace doberman

#endif //DOBERMAN_SRC_LOGGER_SPDLOG_H_
