#ifndef DOBERMAN_SRC_SYNC_AGENT_H_
#define DOBERMAN_SRC_SYNC_AGENT_H_

#include <src/logic/error.h>
#include <src/logic/shared_folder.h>
#include <src/logic/subscribed_folder.h>
#include <src/logic/change_queue.h>
#include <src/logic/subscription.h>
#include <src/logic/envelope_copier.h>
#include <src/logger/logger.h>
#include <src/meta/types.h>
#include <src/service_control/run_status.h>
#include <boost/range.hpp>
#include <boost/system/system_error.hpp>

namespace doberman {
namespace logic {

template <
    typename Subscription,
    typename EnvelopeCopier,
    typename ChangeFilter,
    typename Log,
    typename Profiler>
class Agent {
public:
    using State = SubscriptionState;

    void run() {
        State state = subscription().state();

        logNotice("Start agent", log::subscription_state=state);

        if (state == State::new_) {
            state = init();
        } else if (state == State::init) {
            state = resumeInit();
        }

        while ((state == State::sync || state == State::migrate) && runStatus) {
            const auto change = changes().top();
            if (change) {
                logNotice("Got change", log::change_id=change->id());
                bool isApplicable = false;
                error_code ec;
                std::tie(isApplicable, ec) = applicable(*change);
                if (isApplicable) {
                    ec = change->apply(dst());
                } else if (!ec) {
                    logNotice("Ignore change", log::change_id=change->id());
                }
                if (ec) {
                    failSubscription(ec);
                    return;
                }
                changes().pop();
            } else if (state == State::migrate) {
                subscription().finish();
                return;
            }
            state = subscription().state();
        }

        if (state == State::discontinued) {
            clear();
        } else if (state == State::clear) {
            resumeClear();
        }
    }

    Agent(
            Subscription subscription,
            EnvelopeCopier copier,
            ChangeFilter applicable,
            Log log,
            Profiler profiler,
            const service_control::RunStatus& runStatus)
    : subscription_(std::move(subscription)),
      copier_(std::move(copier)),
      applicable_(std::move(applicable)),
      log_(std::move(log)),
      profiler_(std::move(profiler)),
      runStatus(runStatus) {}

private:
    auto initCopy() {
        const auto handle = subscription().init();
        const auto state = std::get<State>(handle);
        if (state != State::init) {
            return state;
        }

        const auto err = copy(src(), dst());
        if (err) {
            return failSubscription(err);
        }

        return subscription().state();
    }

    auto init() {
        logNotice("initializing");
        const auto state = initCopy();
        if (state != State::init) {
            return state;
        }
        return subscription().sync();
    }

    auto resumeInit() {
        logNotice("resuming initialization");

        auto state = initCopy();
        if (state != State::init) {
            return state;
        }

        const auto replyToRevision = dst().revision();
        while (state == State::init) {
            const auto change = changes().top();
            if (change && replyToRevision >= change->revision()) {
                logNotice("reply change", log::change_id=change->id());
                if (const auto ec = change->apply(dst(), replyToRevision)) {
                    return failSubscription(ec);
                }
                changes().pop();
            } else {
                state = subscription().sync();
            }
        }

        return state;
    }

    auto clearFolder() {
        const auto handle = subscription().clear();
        const auto state = std::get<State>(handle);
        if (state != State::clear) {
            return state;
        }

        const auto start = profiler_.now();
        try {
            dst().clear();
        } catch (const boost::system::system_error& e) {
            return failSubscription(error_code{e.code(), e.what()});
        } catch (const ::std::exception& e) {
            return failSubscription(error_code{error::logic, e.what()});
        }
        profiler_.write("unsubscribe", "clear", profiler_.passed(start));

        return subscription().finish();
    }

    auto clear() {
        logNotice("clearing");
        return clearFolder();
    }

    auto resumeClear() {
        logNotice("resuming clearing");
        return clearFolder();
    }

    auto failSubscription(error_code e) {
        const auto state = subscription().fail(e.message());
        logError(e);
        return state;
    }

    void logError(error_code e) const {
        std::string msg = "Agent: " + e.message();
        DOBBY_LOG_ERROR(log_, msg, log::error_code=e,
                log::subscription_id=this->subscription().id());
    }

    template <typename ... Args>
    void logNotice(std::string message, Args&& ... args) const {
        DOBBY_LOG_NOTICE(log_, std::move(message),
                log::subscription_id=this->subscription().id(),
                std::forward<Args>(args)...);
    }

    auto& subscription() { return detail::dereference(subscription_); }
    auto& subscription() const { return detail::dereference(subscription_); }
    auto& dst() { return subscription().dst(); }
    auto& src() { return subscription().src(); }
    auto& changes() { return subscription().changes();}

    auto applicable(const Change& change) {
        return applicable_(change, dst());
    }

    template <typename SharedFolder, typename SubscribedFolder>
    error_code copy(SharedFolder& src, SubscribedFolder& dst) const {
        const auto start = profiler_.now();
        try {
            copier_(src, dst, [&]{
                auto state = this->subscription().state();
                return state == State::discontinued ||
                       state == State::terminated;
            });
        } catch (const ::boost::system::system_error& e) {
            return error_code{e.code(), e.what()};
        } catch (const ::std::exception& e) {
            return error_code{error::logic, e.what()};
        }
        profiler_.write("init_subscribe", "copy", profiler_.passed(start));
        return error_code{};
    }

    Subscription subscription_;
    EnvelopeCopier copier_;
    ChangeFilter applicable_;
    Log log_;
    Profiler profiler_;
    const service_control::RunStatus& runStatus;
};

struct ChangeFilter {
    template <typename T>
    auto operator()(const Change& change, const SubscribedFolder<T>& dst) const {
        try {
            return std::make_tuple(change.revision() >= dst.revision(), error_code{});
        } catch (const boost::system::system_error& err) {
            return std::make_tuple(false, error_code{err.code(), err.what()});
        }
    }
};

template <
    typename Subscription,
    typename EnvelopeCopier,
    typename ChangeFilter,
    typename Log,
    typename Profiler>
auto makeAgent(Subscription subscription,
                    EnvelopeCopier copier,
                    ChangeFilter applicable,
                    Log log,
                    Profiler profiler,
                    const service_control::RunStatus& runStatus)
{
    return Agent<Subscription, EnvelopeCopier, ChangeFilter, Log, Profiler> {
                std::move(subscription),
                std::move(copier),
                std::move(applicable),
                std::move(log),
                std::move(profiler),
                runStatus };
}

} // namespace logic
} // namespace doberman

#endif /* DOBERMAN_SRC_SYNC_AGENT_H_ */
