#ifndef DOBERMAN_SRC_SYNC_CHANGE_H_
#define DOBERMAN_SRC_SYNC_CHANGE_H_

#include <src/logic/types.h>
#include <macs/envelope_factory.h>
#include <functional>
#include <src/meta/labels.h>

namespace doberman {
namespace logic {

class Change {
public:
    struct LogicSubscribedFolder {
        virtual const SharedFolderCoordinates& coordinates() const = 0;
        virtual void put(EnvelopeWithMimes, meta::labels::LabelsCache) const = 0;
        virtual void erase(macs::MidVec, Revision) const = 0;
        virtual void mark(Mid, std::vector<Label>, Revision) const = 0;
        virtual void unmark(Mid, std::vector<Label>, Revision) const = 0;
        virtual void joinThreads(ThreadId, std::vector<ThreadId>, Revision) const = 0;
        virtual ~LogicSubscribedFolder() = default;
    };

    struct SubscribedFolder {
        SubscribedFolder(const LogicSubscribedFolder& sf, Revision r)
                : subscribedFolder(sf)
                , revision(r) {}

        const Fid& fid() const {
            return subscribedFolder.coordinates().fid;
        }
        void put(EnvelopeWithMimes e, meta::labels::LabelsCache l) const {
            subscribedFolder.put(
                std::make_tuple(
                    macs::EnvelopeFactory(std::get<macs::Envelope>(e))
                        .revision(revision)
                        .release(),
                    std::get<MimeParts>(e)
                ), std::move(l));
        }

        void erase(macs::MidVec m) const {
            subscribedFolder.erase(std::move(m), revision);
        }

        void mark(Mid m, std::vector<Label> l) const {
            subscribedFolder.mark(std::move(m), std::move(l), revision);
        }

        void unmark(Mid m, std::vector<Label> l) const {
            subscribedFolder.unmark(std::move(m), std::move(l), revision);
        }

        void joinThreads(ThreadId tid, std::vector<ThreadId> joinTids) const {
            subscribedFolder.joinThreads(std::move(tid), std::move(joinTids), revision);
        }

    private:
        const LogicSubscribedFolder& subscribedFolder;
        Revision revision;
    };

    using Impl = std::function<error_code(const SubscribedFolder&)>;

    ChangeId id() const { return id_; }

    Revision revision() const { return revision_; }

    error_code apply(const LogicSubscribedFolder& to) const {
        return apply(to, revision());
    }

    error_code apply(const LogicSubscribedFolder& to, Revision r) const {
        try {
            return impl_(SubscribedFolder(to, r));
        } catch(const boost::system::system_error& e) {
            return error_code{e.code(), e.what()};
        }
    }

    Change(ChangeId id, Revision revision, Impl impl)
    : id_(id), revision_(revision), impl_(impl) {}
private:
    ChangeId id_;
    Revision revision_;
    Impl impl_;
};

} // namespace logic
} // namespace doberman

#endif /* DOBERMAN_SRC_SYNC_CHANGE_H_ */
