#pragma once

#include <src/logic/types.h>
#include <src/logic/access.h>
#include <src/logger/logger.h>
#include <boost/optional.hpp>
#include <string>
#include <tuple>

namespace doberman {
namespace logic {

using HostName = std::string;

struct FindJobParams {
    std::chrono::seconds timeout;
    LaunchId launchId;
    HostName hostname;
    std::string workerVersion;
};

template <typename Access, typename RunStatus, typename Log>
class JobFinder {
public:
    JobFinder(Access access, const RunStatus& run, Log log)
    : access_(std::move(access)),
      run_(run),
      actx_(makeContext(access_)),
      log_(std::move(log))
    {}

    boost::optional<Job> find(FindJobParams params) {
        while (run_) {
            try {
                const auto shards = getShards();
                if (shards.empty()) {
                    DOBBY_LOG_WARNING(log_, "no shards found");
                } else if (auto job = findJob(shards, params)) {
                    DOBBY_LOG_NOTICE(log_, "got a job",
                        log::worker_id=std::ref(job->workerId),
                        log::shard_id=std::ref(job->shardId),
                        log::worker_id_time=job->t.time_since_epoch(),
                        log::worker_id_ttl=job->ttl,
                        log::worker_id_state=(outdated(*job) ? "outdated" : "valid"),
                        log::launch_id=std::ref(job->launchId));
                    return job;
                } else {
                    DOBBY_LOG_NOTICE(log_, "no jobs found");
                }
            } catch(const std::exception& e) {
                DOBBY_LOG_ERROR(log_, "got exception and will try one more time",
                        log::where_name="JobFinder::find()", log::exception=e);
            }
        }
        DOBBY_LOG_NOTICE(log_, "jobs search interrupted");
        return boost::none;
    }
private:
    Access access_;
    const RunStatus& run_;
    AccessContext<Access> actx_;
    Log log_;

    auto& access() const { return detail::dereference(access_); }

    auto getShards() {
        return access().getShards(actx_);
    }

    boost::optional<Job> askForJob(const ShardId& name, const FindJobParams& params) {
        return detail::dereference(access_).askForJob(actx_, name, params);
    }

    template <typename Range>
    boost::optional<Job> findJob(Range&& shards, const FindJobParams& params) {
        for (auto& item : shards) {
            try {
                if (auto job = askForJob(item, params)) {
                    return job;
                }
            } catch(const std::exception& e) {
                DOBBY_LOG_ERROR(log_, "got exception and will try next",
                        log::where_name="JobFinder::findJob()", log::exception=e);
            }
        }
        return boost::none;
    }
};


template <typename Access, typename RunStatus, typename Log>
inline auto makeJobFinder(Access access, const RunStatus& run, Log log) {
    return JobFinder<Access, RunStatus, Log>{std::move(access), run, std::move(log)};
}

} // nemaspece logic
} // namespace doberman
