#ifndef DOBERMAN_SRC_SYNC_SUBSCRIBED_FOLDER_H_
#define DOBERMAN_SRC_SYNC_SUBSCRIBED_FOLDER_H_

#include <src/logic/change.h>
#include <src/logic/types.h>
#include <src/meta/labels.h>
#include <src/logic/access.h>

namespace doberman {
namespace logic {

struct MessageCoordinates {
    SharedFolderCoordinates folder;
    Mid mid;
};

template <typename Access>
class SubscribedFolder : public Change::LogicSubscribedFolder {
    Subscriber subscriber_;
    SharedFolderCoordinates coordinates_;
    Access access_;
    AccessContext<Access, Uid> actx_;
    LabelFilter labelFilter_;

    auto messageCoordinates(Mid mid) const {
        return MessageCoordinates{coordinates(), std::move(mid)};
    }
public:
    const Uid& uid() const { return subscriber_.uid; }

    const SharedFolderCoordinates& coordinates() const override { return coordinates_; }

    Revision revision() const {
        return detail::dereference(access_).revision(actx_, coordinates());
    }

    void put(EnvelopeWithMimes e, meta::labels::LabelsCache srcLabelsDict) const override {
        detail::dereference(access_).put(actx_, coordinates(),
                EnvelopeWithMimes(convert(std::get<Envelope>(e), std::move(srcLabelsDict)),
                                  std::get<MimeParts>(e)));
    }

    void initPut(EnvelopeWithMimes e, meta::labels::LabelsCache srcLabelsDict) const {
        detail::dereference(access_).initPut(actx_, coordinates(),
                EnvelopeWithMimes(convert(std::get<Envelope>(e), std::move(srcLabelsDict)),
                                  std::get<MimeParts>(e)));
    }

    void erase(macs::MidVec mids, Revision revision) const override {
        detail::dereference(access_).erase(actx_,
                coordinates(), std::move(mids), revision);
    }

    void mark(Mid mid, std::vector<Label> labels, Revision revision) const override {
        detail::dereference(access_).mark(actx_,
                messageCoordinates(std::move(mid)), convert(labels), revision);
    }

    void unmark(Mid mid, std::vector<Label> labels, Revision revision) const override {
        detail::dereference(access_).unmark(actx_,
                messageCoordinates(std::move(mid)), convert(labels), revision);
    }

    void joinThreads(ThreadId tid, std::vector<ThreadId> joinTids, Revision revision) const override {
        detail::dereference(access_).joinThreads(actx_,
                coordinates(), std::move(tid), std::move(joinTids), revision);
    }

    auto envelopes() const {
        return detail::dereference(access_).envelopes(actx_, coordinates());
    }

    LabelSet labels() const {
        return detail::dereference(access_).labels(actx_);
    }

    void clear() const {
        detail::dereference(access_).clear(actx_, coordinates());
    }

    int64_t lastSyncedImapId() const {
        return detail::dereference(access_).lastSyncedImapId(actx_, coordinates());
    }

    SubscribedFolder(Subscriber s, SharedFolderCoordinates coord, Access acc, LabelFilter labelFilter)
    : subscriber_(std::move(s)), coordinates_(std::move(coord)),
      access_(std::move(acc)), actx_(makeContext(access_, uid())),
      labelFilter_(std::move(labelFilter)) {}

private:
    auto labelCreator() const {
        return [&] (const Label& label) -> Label {
            return detail::dereference(access_).createLabel(actx_, label);
        };
    }

    auto labelFilter() const {
        return [&] (const Label& label) -> bool {
            return labelFilter_.replicable(label);
        };
    }

    std::vector<Label> convert(std::vector<Label> what) const {
        return ::doberman::meta::labels::replicate(labels(), what, labelCreator(), labelFilter());
    }

    Envelope convert(Envelope e, meta::labels::LabelsCache srcDict) const {
        Envelope res;
        std::vector<macs::Lid> notFound;
        const auto dstLabels = labels();
        std::tie(res, notFound) = ::doberman::meta::labels::convertLabels(
                    e, *srcDict, dstLabels, labelCreator(), labelFilter());
        if (!notFound.empty()) {
            std::tie(res, std::ignore) = ::doberman::meta::labels::convertLabels(
                                  e, *(srcDict.update()), dstLabels, labelCreator(), labelFilter());
        }
        return res;
    }
};

template <typename Access>
inline auto makeSubscribedFolder(Subscriber s, SharedFolderCoordinates c, Access a, LabelFilter lf) {
    return SubscribedFolder<Access>(std::move(s), std::move(c), std::move(a), std::move(lf));
}

} // namespace logic
} // namespace doberman

#endif /* DOBERMAN_SRC_SYNC_SUBSCRIBED_FOLDER_H_ */
