#ifndef DOBERMAN_SRC_SYNC_SUBSCRIPTION_H_
#define DOBERMAN_SRC_SYNC_SUBSCRIPTION_H_

#include <src/logic/types.h>
#include <src/logic/change_queue.h>
#include <src/logic/access.h>
#include <src/logic/subscribed_folder.h>
#include <src/logic/shared_folder.h>
#include <macs_pg/subscription/subscription_state.h>

namespace doberman {
namespace logic {

template <
    typename SharedFolderAccess,
    typename SubscribedFolderAccess,
    typename ChangeQueueAccess,
    typename Access,
    typename Handle>
class Subscription {
public:
    using SharedFolder = ::doberman::logic::SharedFolder<SharedFolderAccess>;
    using State = SubscriptionState;
    using SubscribedFolder = ::doberman::logic::SubscribedFolder<SubscribedFolderAccess>;
    using ChangeQueue = ::doberman::logic::ChangeQueue<ChangeQueueAccess>;

    const SubscriptionId& id() const {return handle().id;}
    SharedFolder& src() { return src_; }
    SubscribedFolder& dst() { return dst_; }
    ChangeQueue& changes() { return changes_;}

    State state() const { return access().state(actx_); }

    auto init() const { return access().init(actx_); }

    State sync() const { return access().sync(actx_); }

    State fail(std::string message) const { return access().fail(actx_, std::move(message)); }

    State finish() const { return access().finish(actx_);}

    auto clear() const { return access().clear(actx_); }

    Subscription(
            Handle h,
            SharedFolderAccess src,
            SubscribedFolderAccess dst,
            ChangeQueueAccess changes,
            Access access,
            LabelFilter labelFilter)
    : handle_(std::move(h)),
      src_(makeSharedFolder(handle().folder, std::move(src))),
      dst_(makeSubscribedFolder(handle().subscriber, handle().folder, std::move(dst), std::move(labelFilter))),
      changes_(makeChangeQueue(id(), std::move(changes))),
      access_(std::move(access)),
      actx_(makeContext(access_, id(), handle().subscriber)){
    }

private:
    auto& handle() const { return detail::dereference(handle_); }
    auto& access() const { return detail::dereference(access_); }

    Handle handle_;
    SharedFolder src_;
    SubscribedFolder dst_;
    ChangeQueue changes_;
    Access access_;
    AccessContext<Access, SubscriptionId, Subscriber> actx_;
};

template <
        typename SharedFolderAccess,
        typename SubscribedFolderAccess,
        typename ChangeQueueAccess,
        typename Access,
        typename Handle>
auto makeSubscription(Handle handle,
                      SharedFolderAccess src,
                      SubscribedFolderAccess dst,
                      ChangeQueueAccess changes,
                      Access access,
                      LabelFilter labelFilter) {
    return Subscription<SharedFolderAccess,
                        SubscribedFolderAccess,
                        ChangeQueueAccess,
                        Access,
                        Handle>(
            std::move(handle),
            std::move(src),
            std::move(dst),
            std::move(changes),
            std::move(access),
            std::move(labelFilter));
}

} // namespace logic
} // namespace doberman

#endif /* DOBERMAN_SRC_SYNC_SUBSCRIPTION_H_ */
