#pragma once

#include <src/logic/access.h>
#include <src/logic/subscription.h>
#include <boost/range/adaptors.hpp>
#include <boost/range/algorithm_ext/push_back.hpp>
#include <boost/range/algorithm.hpp>
#include <memory>


namespace doberman {
namespace logic {

template <typename Access>
class SubscriptionRepository {
    using Data = SubscriptionData;
    using DataMap = std::map<SubscriptionId, Data>;

    struct Ctx;

    class Handle {
        class Deleter {
            std::tuple<DataMap::iterator, Ctx*> v_;

        public:
            Deleter(DataMap::iterator i = {}, Ctx* ctx = nullptr) : v_(i, ctx) {}

            void operator() (Data* data) const { release(data); }

            void release(Data* data) const {
                if (data) {
                    access().release(actx(), id());
                    erase();
                }
            }

            void decline(Data* data, std::string reason) const {
                if (data) {
                    access().decline(actx(), id(), std::move(reason));
                    erase();
                }
            }

        private:
            auto iterator() const { return std::get<DataMap::iterator>(v_);}
            auto& access() const { return doberman::detail::dereference(std::get<Ctx*>(v_)->access_);}
            auto& actx() const { return std::get<Ctx*>(v_)->actx_;}
            auto& id() const { return iterator()->second.id;}
            void erase() const { std::get<Ctx*>(v_)->data_.erase(iterator()); }
        };

        std::unique_ptr<Data, Deleter> v;
    public:
        Handle() = default;
        Handle(DataMap::iterator i, Ctx& ctx)
        : v(std::addressof(i->second), Deleter(i, std::addressof(ctx))) {}

        auto operator -> () const { return v.get();}
        auto& operator * () const { return *v;}
        friend auto& dereference(Handle& h) { return *h;}
        friend auto& dereference(const Handle& h) { return *h;}
        void release() { v.reset(); }
        void decline(std::string reason) {
            v.get_deleter().decline(v.release(), std::move(reason));
        }
        operator bool () const noexcept { return bool(v); }
    };

public:
    SubscriptionRepository(Access access, int maxCount)
    : ctx_(std::make_unique<Ctx>(std::move(access), maxCount)) {
        fetchReserved();
        releaseOverdraft();
    }

    Handle get() {
        if (queue().empty()) {
            enqueue(access().reserve(actx(), credit()));
        }

        return queue().empty() ? Handle{} : pop();
    }

    static void release(Handle h) {
        h.release();
    }

    static void decline(Handle h, std::string reason) {
        h.decline(std::move(reason));
    }

    int maxCount() const noexcept {
        return ctx_->maxCount_;
    }

    int count() const noexcept {
        return int(data().size());
    }

    int credit() const noexcept {
        return count() < maxCount() ? maxCount() - count() : 0;
    }

    int overdraft() const noexcept {
        return count() > maxCount() ? count() - maxCount() : 0;
    }

private:
    Handle pop() {
        auto retval = std::move(queue().front());
        queue().pop();
        return retval;
    }

    void releaseOverdraft() {
        for (; overdraft() && !queue().empty(); release(pop()));
    }

    void fetchReserved() {
        enqueue(access().getReserved(actx()));
    }

    template <typename Range>
    void enqueue(Range&& src) {
        const auto store = [&] (const Data& v) {
            const auto res = data().emplace(v.id, v);
            return res.second ? res.first : data().end();
        };

        boost::for_each(src, [&](auto& v) {
            const auto stored = store(v);
            if (stored == this->data().end()) {
                throw std::invalid_argument("SubscriptionRepository::enqueue() invalid iterator");
            }
            this->queue().push(Handle(stored, *ctx_));
        });
    }

    struct Ctx {
        Access access_;
        AccessContext<Access> actx_;
        DataMap data_;
        std::queue<Handle> queue_;
        int maxCount_;
        Ctx(Access access, int maxCount)
        : access_(std::move(access)),
          actx_(makeContext(access_)), maxCount_(maxCount){}
    };

    std::unique_ptr<Ctx> ctx_;
    auto& access() { return detail::dereference(ctx_->access_); }
    auto& queue() { return ctx_->queue_; }
    auto& data() { return ctx_->data_; }
    const auto& data() const { return ctx_->data_; }
    auto& actx() { return ctx_->actx_; }
};

template <typename Access>
auto makeSubscriptionRepository(Access access, int maxCount) {
    return SubscriptionRepository<Access>{std::move(access), maxCount};
}

} // namespace logic
} // namespace doberman
