#include <iostream>

#include <src/meta/pg.h>
#include <src/meta/http_client.h>
#include <src/access/spawn.h>
#include <src/access/factory.h>
#include <src/access_impl/factory.h>
#include <src/access_impl/job_finder.h>
#include <src/access_impl/heartbeat.h>
#include <src/access_impl/subscription_resource.h>
#include <src/service_control/service_control.h>
#include <src/service_control/command_source_impl.h>
#include <src/profiling/pa_profiler.h>
#include <sharpei_client/sharpei_client.h>
#include <boost/numeric/conversion/cast.hpp>

using namespace doberman;
using namespace service_control;

int main(int argc, char* argv[]) {
    if (argc < 2) {
        std::cerr << "Dobby wants config file" << std::endl;
        return 1;
    }

    const std::string workerVersion = argc == 3 ? argv[2] : "";

    auto cfg = ::meta::conf::fromYamlFile(std::string(argv[1]));

    auto logger = log::spd::make_log(cfg.log);
    auto log = std::get<0>(logger);
    auto sink = std::get<1>(logger);

    meta::http::initYLog(cfg.ylog);

    auto profiler = profiling::pa::makeProfiler(cfg.profiler_log_file);

    const auto launchId = LaunchId{};

    DOBBY_LOG_NOTICE(log, "Dobby is starting", log::launch_id=launchId);
    try {
        LabelFilter labelFilter(cfg.not_replicable_labels);

        RunStatus runStatus;

        boost::asio::io_service ios;

        auto pgPool = meta::createPool(cfg.pool, ios);

        boost::optional<Job> job;

        boost::asio::spawn(ios, [&](auto yield) {
            const auto controlProto = cfg.control.use_ipv6 ? boost::asio::ip::tcp::v6()
                                                           : boost::asio::ip::tcp::v4();
            boost::asio::ip::tcp::endpoint endpoint(controlProto, cfg.control.port);
            auto commandSource = makeCommandSourceImpl(ios, runStatus, log, endpoint);
            auto control = makeServiceControl(ios, log, *sink, commandSource, runStatus, statPublisher(*pgPool, job));
            control.run(yield);
        });

        auto httpPool = meta::http::makePool(ios, cfg.sharpei.max_requests);

        logic::FindJobParams findJobParams{
            std::chrono::seconds(cfg.doberman.worker_id_lease_timeout_s),
            launchId,
            boost::asio::ip::host_name(),
            workerVersion
        };

        boost::asio::spawn(ios, [&](auto yield) {
            try{
                auto factory = meta::createFactory(cfg, pgPool, httpPool, ios, log);

                auto sclient = sharpei::client::createSharpeiClient(
                        std::make_shared<meta::http::SharpeiClient>(httpPool, ios),
                        meta::makeSharpeiParams(cfg.sharpei, log),
                        sharpei::client::RequestInfo{"","","dobby","",""});

                auto finder = logic::makeJobFinder(
                        access::makeJobFinder(
                            access_impl::makeJobFinder(
                                    std::move(sclient),
                                    [factory](ShardId id){ return factory->shard(id);},
                                    std::chrono::seconds(cfg.doberman.job_discovery_timeout_s)),
                            yield),
                        runStatus, log);
                job = finder.find(findJobParams);
                if (job) {
                    boost::asio::spawn(ios, access_impl::makeHeartbeat(
                            factory->shard(job->shardId),
                            runStatus, WorkerIdControl(*job), log, Seconds{1}));

                    auto accessFactoryImpl = access_impl::makeAccessFactory(
                        std::move(factory),
                        *job,
                        profiler,
                        log,
                        cfg.sleep_times,
                        cfg.retries,
                        access_impl::makeSubscriptionResource(ios,
                                cfg.doberman.max_subscription_heavy_process_count,
                                cfg.doberman.max_subscriptions_count),
                        cfg.doberman.envelope_chunk_size
                    );

                    access::spawn(
                            ios,
                            boost::numeric_cast<int>(cfg.doberman.max_subscriptions_count),
                            access::makeAccessFactory(std::move(accessFactoryImpl)),
                            log,
                            profiler,
                            labelFilter,
                            runStatus);
                }
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const std::exception& e) {
                DOBBY_LOG_ERROR_WHERE(log, "unexpected exception: ", log::exception=e);
            } catch (...) {
                DOBBY_LOG_ERROR_WHERE(log, "unexpected unknown exception");
            }
        });

        ios.run();
    } catch (const std::exception& e) {
        std::cerr << "main: unexpected std::exception:" << e.what() << std::endl;
        DOBBY_LOG_ERROR(log, "main: unexpected exception: ", log::exception=e);
        return 1;
    } catch (...) {
        std::cerr << "main: unexpected unknown exception" << std::endl;
        DOBBY_LOG_ERROR(log, "main: unexpected unknown exception");
        return 1;
    }
    return 0;
}
