#ifndef DOBERMAN_SRC_META_CHANGED_H_
#define DOBERMAN_SRC_META_CHANGED_H_

#include <cstdint>
#include <string>
#include <vector>
#include <boost/optional.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <boost/range/join.hpp>

#include <yamail/data/deserialization/json_reader.h>
#include <macs_pg/changelog/change.h>
#include <src/meta/types.h>
#include <src/meta/labels.h>


BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(changed)(detail), Mid,
        (std::int64_t, mid)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(changed), MoveArguments,
        (std::int32_t, fid)
)

using MdbLids = std::vector<std::int32_t>;
using MdbFlag = boost::optional<bool>;

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(changed), UpdateArguments,
        (MdbFlag, seen)
        (MdbFlag, recent)
        (MdbFlag, deleted)
        (MdbLids, lids_add)
        (MdbLids, lids_del)
)

using MdbTid = std::int64_t;
using MdbTidVec = std::vector<MdbTid>;

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(changed), JoinThreadsArguments,
        (MdbTid, tid)
        (MdbTidVec, join_tids)
)

namespace doberman {
namespace meta {
namespace changed {
namespace detail {


using ChangeType = ::macs::pg::ChangeType;
using yamail::data::deserialization::fromJson;

inline bool isStoreChange(const ChangeType& type) {
    return type == ChangeType::store ||
        type == ChangeType::copy ||
        type == ChangeType::syncStore;
}

inline bool isEraseChange(const ChangeType& type) {
    return type == ChangeType::delete_ ||
        type == ChangeType::syncDelete;
}

inline bool isUpdateChange(const ChangeType& type) {
    return type == ChangeType::update ||
        type == ChangeType::syncUpdate;
}

inline bool isJoinThreadsChange(const ChangeType& type) {
    return type == ChangeType::threadsJoin ||
        type == ChangeType::syncThreadsJoin;
}

inline auto filterLids2Labels(MdbLids argsLids, const LabelSet& labelsDict) {
    using namespace ::boost::adaptors;

    const auto i2s = [](MdbLids::value_type l){return std::to_string(l);};
    auto lids = argsLids | transformed(i2s);
    auto labels = labels::lids2Labels(labelsDict, lids);
    const auto& foundLabels = std::get<0>(labels);

    std::vector<::macs::Label> res;
    boost::copy(foundLabels | indirected | map_values, std::back_inserter(res));

    std::vector<::macs::Lid> notFound;
    boost::copy(std::get<1>(labels) | indirected, std::back_inserter(notFound));

    return std::make_tuple(std::move(res), std::move(notFound));
}

} // namespace detail


template <typename ArgsStruct>
inline auto getArguments(const ::macs::Change& change) {
    if (!change.arguments()) {
        throw std::domain_error(
            std::string(__PRETTY_FUNCTION__) +
            " got change without arguments, id:" + std::to_string(change.changeId())
        );
    }
    return detail::fromJson<ArgsStruct>(*change.arguments());
}

enum class ChangeAction:char {
    put = 'p',
    erase = 'e',
    update = 'u',
    joinThreads = 'j',
    move = 'm',
};

inline auto getMidsFromChange(const ::macs::Change& change) {
    using ChangedMids = std::vector<detail::Mid>;
    if (!change.changed()) {
        throw std::domain_error(
            std::string(__PRETTY_FUNCTION__) +
            " got change without .changed, change_id:" + std::to_string(change.changeId()) + " !");
    }
    auto changedEntries = detail::fromJson<ChangedMids>(*(change.changed()));
    if (changedEntries.empty()) {
        /* unlikely case, but we don't have strict guaranties in database */
        throw std::domain_error(
            std::string(__PRETTY_FUNCTION__) +
            " got change with empty changed, id: " + std::to_string(change.changeId())
        );
    }
    ::macs::MidList res;
    std::transform(changedEntries.cbegin(), changedEntries.cend(), std::back_inserter(res),
        [](const auto& ce){return std::to_string(ce.mid);});
    return res;
}

inline auto getUpdatedLabels(const ::macs::Change& change, const LabelSet& labelsDict) {
    auto args = getArguments<UpdateArguments>(change);

    std::vector<::macs::Label> labelsAdd, labelsDel;
    std::vector<::macs::Lid> notFoundAdd, notFoundDel;
    std::tie(labelsAdd, notFoundAdd) = detail::filterLids2Labels(args.lids_add, labelsDict);
    std::tie(labelsDel, notFoundDel) = detail::filterLids2Labels(args.lids_del, labelsDict);

    using Symbol = ::macs::Label::Symbol;
    const auto seq = std::make_tuple(
                std::make_tuple(args.seen, Symbol::seen_label),
                std::make_tuple(args.recent, Symbol::recent_label),
                std::make_tuple(args.deleted, Symbol::deleted_label));
    boost::fusion::for_each(seq, [&](auto& item) {
        if (auto f = std::get<MdbFlag>(item)) {
            (*f ? labelsAdd : labelsDel).push_back(labelsDict.at(std::get<Symbol>(item)));
        }
    });

    auto notFound = boost::range::join(notFoundAdd, notFoundDel);
    return std::make_tuple(std::move(labelsAdd), std::move(labelsDel),
                           std::vector<::macs::Lid>{std::begin(notFound), std::end(notFound)});
}

inline auto getJoinedThreads(const ::macs::Change& change) {
    auto args = getArguments<JoinThreadsArguments>(change);

    ThreadId tid = std::to_string(args.tid);
    std::vector<ThreadId> joinTids;
    joinTids.reserve(args.join_tids.size());
    std::transform(args.join_tids.begin(), args.join_tids.end(), std::back_inserter(joinTids),
                   [](const auto& t){ return std::to_string(t); });

    return std::make_pair(std::move(tid), std::move(joinTids));
}

inline ChangeAction getActionByChange(const macs::Change& change) {
    if (detail::isStoreChange(change.type())) {
        return ChangeAction::put;
    } else if (detail::isEraseChange(change.type())) {
        return ChangeAction::erase;
    } else if (detail::isUpdateChange(change.type())) {
        return ChangeAction::update;
    } else if (detail::isJoinThreadsChange(change.type())) {
        return ChangeAction::joinThreads;
    } else if (change.type() == detail::ChangeType::move) {
        return ChangeAction::move;
    }
    throw std::logic_error(
        std::string(__PRETTY_FUNCTION__) +
        "unsupported change type: " + change.type().toString() + ", id:" + std::to_string(change.changeId())
    );
}

} // namespace changed
} // namespace meta
} // namespace doberman


#endif /* DOBERMAN_SRC_META_CHANGED_H_ */
