#ifndef DOBERMAN_SRC_META_HTTP_CLIENT_H_
#define DOBERMAN_SRC_META_HTTP_CLIENT_H_

#include <ostream>
#include <sharpei_client/http.h>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

#include <ymod_httpclient/client.h>
#include <ymod_httpclient/errors.h>

#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <src/logger/spdlog.h>
#include <boost/algorithm/string/join.hpp>
#include <yamail/resource_pool.hpp>

namespace doberman {
namespace meta {
namespace http {

using pool = ::yamail::resource_pool::async::pool<std::shared_ptr<yhttp::client>>;

class SharpeiClient : public sharpei::client::http::HttpClient {
public:
    explicit SharpeiClient(pool& client, boost::asio::io_service& service)
        : taskContext(boost::make_shared<yplatform::task_context>()),
          client(client),
          service(service)
    {}

    void aget(const sharpei::client::http::Address& addr,
        sharpei::client::http::Timeout timeout,
        const std::string& method,
        const sharpei::client::http::Arguments& args,
        const sharpei::client::http::Headers& /*headers*/,
        sharpei::client::http::ResponseHandler handler,
        bool /*keepAlive*/,
        const std::string& /*requestId*/) const override {

        using ShrapeiResponse = sharpei::client::http::Response;

        yhttp::options opts;
        //opts.reuse_connection = true;
        opts.timeouts.total = timeout;

        client.get_auto_recycle(service,
                                [handler = std::move(handler),
                                 request=makeGetRequest(addr, method, args),
                                 taskContext = this->taskContext,
                                 opts]
                 (auto err, auto handle) mutable {
            if (err) {
                return handler(err, ShrapeiResponse());
            }

            handle.get()->async_run(
                    taskContext,
                    std::move(request),
                    opts,
                    [handler = std::move(handler)](auto ec, auto response) {
                if (ec) {
                    handler(ec, ShrapeiResponse());
                } else {
                    ShrapeiResponse resp {
                        static_cast<unsigned>(response.status),
                        std::move(response.body)
                    };
                    handler({}, std::move(resp));
                }
            });
        }, yamail::resource_pool::time_traits::duration::max());
    }

    void apost(const sharpei::client::http::Address&,
        sharpei::client::http::Timeout,
        const std::string&,
        const sharpei::client::http::Arguments&,
        const sharpei::client::http::Headers&,
        const std::string&,
        sharpei::client::http::ResponseHandler,
        bool,
        const std::string&) const override {

        throw std::logic_error( std::string(__PRETTY_FUNCTION__) + " not implemented!");
    }
private:
    using Client = yhttp::client;
    using TaskContextPtr = Client::task_context_ptr;

    static yhttp::request makeGetRequest(const sharpei::client::http::Address& addr,
                                                   const std::string& method,
                                                   const sharpei::client::http::Arguments& args) {
        std::ostringstream url;
        url << addr.host;
        url << ":" << addr.port;
        url << method;

        if (!args.empty()) {
            url << "?";
        }

        const auto foldArg = [](auto& arg) {
            return arg.first + "=" + arg.second.front();
        };

        url << boost::algorithm::join(args | boost::adaptors::transformed(foldArg), "&");

        return yhttp::request::GET(url.str());
    }

    TaskContextPtr taskContext;
    pool& client;
    boost::asio::io_service& service;
};

inline auto makePool(boost::asio::io_service& ios, std::size_t capacity) {
    ymod_httpclient::settings cs;
    cs.enable_logging = true;
    cs.enable_headers_logging = true;
    cs.user_agent = "doberman";

    const auto client = std::make_shared<yhttp::client>(ios, cs);
    return pool([client]{return client;}, capacity, 10000000);
}

inline void initYLog(const log::spd::SpdlogConfig& ylog) {
    auto ysink = log::spd::makeSink(ylog);
    auto ylogger = log::spd::makeSpdLog(ylog, "global", ysink);

    yplatform::log::global_log_init({ylogger}, {ysink});
}

} // namespace http
} // namespace meta
} // namespace doberman

#endif /* DOBERMAN_SRC_META_HTTP_CLIENT_H_ */
