#ifndef DOBERMAN_SRC_META_LABELS_H_
#define DOBERMAN_SRC_META_LABELS_H_

#include <macs/label_set.h>
#include <macs/envelope_factory.h>
#include <macs/label_factory.h>
#include <boost/range.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/range/combine.hpp>
#include <boost/move/algorithm.hpp>
#include <boost/assign.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <set>
#include <functional>

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(labels), NotReplicableLabels,
        (std::vector<std::string>, symbols)
        (std::vector<std::string>, types)
)

namespace doberman {
namespace meta {
namespace labels {

class LabelFilter {
public:
    LabelFilter(const NotReplicableLabels& cfg) {
        using namespace boost::adaptors;
        boost::copy(cfg.symbols | transformed(::macs::Label::Symbol::getByTitle),
                    std::inserter(symbols_, symbols_.end()));
        boost::copy(cfg.types | transformed(::macs::Label::Type::getByTitle),
                    std::inserter(types_, types_.end()));
    }

    bool replicable(const ::macs::Label& label) const {
        bool inFilter = hasSymbol(label.symbolicName()) ||
                        hasType(label.type());
        return !inFilter;
    }

private:
    std::set<::macs::Label::Symbol> symbols_;
    std::set<::macs::Label::Type> types_;

    bool hasSymbol(const ::macs::Label::Symbol& symbol) const {
        return symbols_.count(symbol);
    }

    bool hasType(const ::macs::Label::Type& type) const {
        return types_.count(type);
    }
};

class LabelsCache {
    using FetchHandler = std::function<::macs::LabelSet()>;
public:
    LabelsCache(const ::macs::LabelSet& l, FetchHandler f)
        : labels(l)
        , fetch(std::move(f)) {}

    ::macs::LabelSet& operator*() { return labels; }
    const ::macs::LabelSet& operator*() const { return labels; }

    LabelsCache& update() {
        labels = fetch();
        return *this;
    }

private:
    ::macs::LabelSet labels;
    FetchHandler fetch;
};

template <typename RangeOfLids>
inline auto lids2Labels(const ::macs::LabelSet& labels, RangeOfLids& lids) {
    std::vector<decltype(std::begin(lids))> notFoundIters;
    std::vector<decltype(std::begin(labels))> foundIters;
    for (auto lid=std::begin(lids); lid!=std::end(lids); ++lid) {
        const auto i = labels.find(*lid);
        if (i != labels.end()) {
            foundIters.push_back(i);
        } else {
            notFoundIters.push_back(lid);
        }
    }

    return std::make_tuple(std::move(foundIters), std::move(notFoundIters));
}

namespace detail {

inline bool hasSymbol(const ::macs::Label& l) {
    return l.symbolicName() != ::macs::Label::Symbol::none;
}

inline auto findSimilar(const ::macs::LabelSet& in, const ::macs::Label& to) {
    return hasSymbol(to) ? in.find(to.symbolicName()) : in.find(to.name(), to.type());
}

template <typename RangeOfLabels, typename LabelCreator, typename LabelFilter>
inline auto replicate(const ::macs::LabelSet& in, RangeOfLabels&& what,
                      LabelCreator createSimilar, LabelFilter replicable) {
    using namespace ::boost::adaptors;
    return what | filtered(replicable) | transformed([&in, createSimilar] (const ::macs::Label& arg) {
            const auto i = findSimilar(in, arg);
            if( i == in.end() ) {
                return createSimilar(arg);
            }
            return i->second;
        });
}

} // namespace detail

template <typename RangeOfLabels, typename LabelCreator, typename LabelFilter>
inline std::vector<::macs::Label> replicate(const ::macs::LabelSet& in,
        RangeOfLabels&& what, LabelCreator createSimilar, LabelFilter replicable) {
    auto rng = detail::replicate(in, what, std::move(createSimilar), std::move(replicable));
    return {std::begin(rng), std::end(rng)};
}

template <typename LabelCreator, typename LabelFilter>
inline auto convertLabels(::macs::Envelope forEnvelope,
                        const ::macs::LabelSet& srcDict,
                        const ::macs::LabelSet& dstDict,
                        LabelCreator createSimilar,
                        LabelFilter replicable) {

    auto converted = lids2Labels(srcDict, forEnvelope.labels());
    const auto& labels = std::get<0>(converted);

    using namespace ::boost::adaptors;
    auto replicas = detail::replicate(dstDict, labels | indirected | map_values, createSimilar, replicable);

    ::macs::EnvelopeFactory f{forEnvelope};
    f.clearLabels();

    for(const auto& label : replicas) {
        f.addLabelID(label.lid());
    };

    auto notFound = std::get<1>(converted) | indirected;
    return std::make_tuple(f.release(), std::vector<::macs::Lid>{std::begin(notFound), std::end(notFound)});
}

} // namespace labels
} // namespace meta

using meta::labels::LabelFilter;
} // namespace doberman



#endif /* DOBERMAN_SRC_META_LABELS_H_ */
