#ifndef DOBERMAN_SRC_META_PG_H_
#define DOBERMAN_SRC_META_PG_H_

#include <fstream>
#include <macs_pg/macs_pg.h>

#include <src/meta/http_client.h>
#include <src/logger/logger.h>
#include <src/access_impl/factory.h>

#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/deserialization/yaml.h>
#include <yamail/data/serialization/yajl.h>

#include <macs_pg/integration/pa_profiler.h>


#include <boost/fusion/adapted/struct/define_struct.hpp>


BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Pool,
        (std::int64_t, connect_timeout_ms)
        (std::int64_t, queue_timeout_ms)
        (std::int64_t, query_timeout_ms)
        (std::int64_t, max_connections)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Sharpei,
        (std::string, host)
        (unsigned, port)
        (unsigned, retries)
        (std::int64_t, timeout_ms)
        (std::size_t, max_requests)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Database,
        (std::int64_t, transaction_timeout_ms)
        (std::string, query_conf)
        (std::string, user)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Control,
        (unsigned short, port)
        (bool, use_ipv6)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Doberman,
        (unsigned, max_subscriptions_count)
        (unsigned, max_subscription_heavy_process_count)
        (unsigned, job_discovery_timeout_s)
        (unsigned, worker_id_lease_timeout_s)
        (unsigned, envelope_chunk_size)
)

BOOST_FUSION_DEFINE_STRUCT((doberman)(meta)(conf), Cfg,
        (doberman::meta::conf::Pool, pool)
        (doberman::meta::conf::Sharpei, sharpei)
        (doberman::meta::conf::Database, database)
        (doberman::meta::conf::Doberman, doberman)
        (doberman::log::spd::SpdlogConfig, log)
        (doberman::log::spd::SpdlogConfig, ylog)
        (doberman::meta::conf::Control, control)
        (std::string, profiler_log_file)
        (doberman::access_impl::SleepTimes, sleep_times)
        (doberman::access_impl::Retries, retries)
        (doberman::meta::labels::NotReplicableLabels, not_replicable_labels)
)

BOOST_FUSION_ADAPT_STRUCT(macs::pg::statistic::ConnectionStats,
        freeConnections,
        busyConnections,
        pendingConnections,
        maxConnections,
        queueSize,
        droppedTimedOut,
        droppedFailed,
        droppedBusy,
        droppedWithResult
)

BOOST_FUSION_DEFINE_STRUCT((doberman), JobInfo,
        (macs::WorkerId, worker_id)
        (std::string, launch_id)
        (doberman::ShardId, shard_id)
)

BOOST_FUSION_DEFINE_STRUCT((doberman), Statistics,
        (macs::pg::statistic::ConnectionPoolStats, pool)
        (doberman::JobInfo, job)
)

namespace pgg {
template <typename T>
inline decltype(auto) to_string(const pgg::Enumeration<T>& v) {
    return v.toString();
}

}

namespace doberman {
namespace meta {
namespace conf {

using boost::property_tree::ptree;

inline auto fromPtree(ptree& p) {
    return ::yamail::data::deserialization::fromPtree<Cfg>(p);
}

inline auto fromYamlFile(std::string path) {
    std::ifstream istrm(path);
    if (!istrm.is_open()) {
        throw std::runtime_error(
            std::string(__PRETTY_FUNCTION__) +
            " failed to open config " + path);
    }
    auto ret = ::yamail::data::deserialization::fromYaml<Cfg>(istrm);
    istrm.close();
    return ret;
}

} // namespace conf

template <typename Logger>
inline auto makeSharpeiParams(const conf::Sharpei& cfg, Logger logger) {
    using ::sharpei::client::http::Timeout;
    ::sharpei::client::Settings s;
    s.sharpeiAddress = {cfg.host, cfg.port};
    s.timeout = Timeout(cfg.timeout_ms);
    s.retries = cfg.retries;
    s.keepAlive = false;
    s.onHttpException = [logger] (const std::exception& e) {
        DOBBY_LOG_ERROR(logger, "sharpei client http client exception", log::exception=e);
    };
    s.onHttpError = [logger] (const mail_errors::error_code& e, const std::string& msg) {
        DOBBY_LOG_ERROR(logger, "sharpei client http client error: " + msg, log::error_code=e);
    };
    return s;
}

inline auto createPool(const conf::Pool& cfg, boost::asio::io_service& ios) {
    using Ms = macs::pg::Milliseconds;
    return macs::pg::ConnectionPoolFactory()
            .ioService(ios)
            .connectTimeout(Ms(cfg.connect_timeout_ms))
            .queueTimeout(Ms(cfg.queue_timeout_ms))
            .queryTimeout(Ms(cfg.query_timeout_ms))
            .maxConnections(static_cast<std::size_t>(cfg.max_connections))
            .product();
}

template <typename Logger>
inline auto createFactory(
                const conf::Cfg& cfg,
                macs::pg::ConnectionPoolPtr pool,
                http::pool& client,
                boost::asio::io_service& service,
                Logger logger ) {

    using Ms = macs::pg::Milliseconds;
    auto queryConf = macs::pg::readQueryConfFile(cfg.database.query_conf);

    auto clientPtr = std::make_shared<http::SharpeiClient>(client, service);
    auto sharpeiParams = makeSharpeiParams(cfg.sharpei, logger);
    auto uidResolver = macs::pg::createSharpeiUidResolverFactory(
            {clientPtr, sharpeiParams, {cfg.database.user, ""}});
    auto shardResolver = macs::pg::createShardResolverFactory(
            {clientPtr, sharpeiParams, {cfg.database.user, ""}});

    auto factory = macs::pg::createSeviceFactory(
            pool,
            std::move(uidResolver),
            std::move(shardResolver));

    factory->queryConf(queryConf)
            .credentials({cfg.database.user, ""})
            .transactionTimeout(Ms(cfg.database.transaction_timeout_ms))
            .logger(::pgg::logging::makeTypedLog(log::bind(logger, log::message="macs_pg")))
            .profiler(::macs::pg::integration::PaProfiler::create());

    return std::shared_ptr<::macs::pg::ServiceFactory>(factory.release());
}

} // namespace meta

inline auto statPublisher(const macs::pg::database::ConnectionPool& pool, const boost::optional<Job>& job) {
    return [&pool, &job]{
        Statistics stat;
        stat.pool = macs::pg::statistic::getPoolStats(pool);
        if (job) {
            stat.job.worker_id = job->workerId;
            stat.job.launch_id = to_string(job->launchId);
            stat.job.shard_id = job->shardId;
        }
        std::ostringstream s;
        yamail::data::serialization::writeJson(s, stat) << std::endl;
        return s.str();
    };
}

} // namespace doberman

#endif /* DOBERMAN_SRC_META_PG_H_ */
