#ifndef DOBERMAN_SRC_META_TYPES_H_
#define DOBERMAN_SRC_META_TYPES_H_

#include <chrono>

#include <macs/types.h>
#include <macs/revision.h>
#include <macs/io.h>
#include <macs/threads_meta.h>
#include <macs/hooks.h>
#include <macs_pg/subscription/types.h>
#include <macs_pg/subscription/subscription_state.h>
#include <macs_pg/subscribed_folders/sync_message.h>
#include <yplatform/tskv/tskv.h>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <boost/uuid/random_generator.hpp>

namespace doberman {

using SubscriptionState = ::macs::pg::SubscriptionState;
using Uid = std::string;
using Revision = ::macs::Revision;
using yield_context = decltype(::io_result::make_yield_context(std::declval<boost::asio::yield_context>()));
using Mid = ::macs::Mid;
using Fid = ::macs::Fid;

struct SubscriptionId {
    Uid uid;
    ::macs::SubscriptionId id = 0;
    SubscriptionId() = default;
    SubscriptionId(Uid uid, ::macs::SubscriptionId id) : uid(uid), id(id) {}
};

template <typename Out>
inline void to_tskv(Out& out, const char* key, const SubscriptionId& v) {
    const std::string name = key;
    out << ytskv::attr(name + ".uid", v.uid)
        << ytskv::attr(name + ".id", v.id);
}

inline bool operator == (const SubscriptionId& lhs, const SubscriptionId& rhs) {
    return lhs.uid == rhs.uid && lhs.id == rhs.id;
}

inline bool operator < (const SubscriptionId& lhs, const SubscriptionId& rhs) {
    return std::tie(lhs.uid, lhs.id) < std::tie(rhs.uid, rhs.id);
}

using Envelope = ::macs::Envelope;
using Label = ::macs::Label;
using LabelSet = ::macs::LabelSet;
using ChangeId = std::int64_t;
using error_code = ::macs::error_code;
using system_error = ::macs::system_error;

using MimeParts = ::macs::MimeParts;
using SyncMessage = ::macs::SyncMessage;

using EnvelopeWithMimes = std::tuple<Envelope, MimeParts>;

using Seconds = std::chrono::seconds;
using ThreadId = ::macs::ThreadId;

class LaunchId {
    boost::uuids::uuid id;
public:
    template <typename Generator = boost::uuids::random_generator>
    explicit LaunchId(Generator g = Generator()) : id(g()) {}
    const boost::uuids::uuid& uuid() const { return id;}
    friend std::string to_string(const LaunchId& v) {
        return boost::uuids::to_string(v.uuid());
    }
};


using ShardId = std::string;

struct Job {
    ::macs::WorkerId workerId;
    std::chrono::steady_clock::duration ttl;
    std::chrono::steady_clock::time_point t;
    LaunchId launchId;
    ShardId shardId;
    friend bool outdated(const Job& v) {
        return std::chrono::steady_clock::now() - v.t > v.ttl;
    }
};


class WorkerId {
public:
    WorkerId() = default;
    WorkerId(const Job& job) : job_(std::addressof(job)){}
    ::macs::WorkerId id() const { return job_ ? job_->workerId : ::macs::WorkerId{};}
    operator ::macs::WorkerId () const { return id();}
    friend std::string to_string(const WorkerId& v) {
        return v.id();
    }
    bool valid() const { return job_ ? !outdated(*job_) : false; }
    auto launchId() const { return job_ ? to_string(job_->launchId) : std::string(); }

    bool operator == (const WorkerId& other) const {
        return job_ == other.job_;
    }

    bool operator != (const WorkerId& other) const {
        return !(*this == other);
    }
private:
    const Job* job_ = nullptr;
};

class WorkerIdControl {
    Job& job_;
public:
    WorkerIdControl(Job& data) : job_(data) {}
    ::macs::WorkerId id() const { return job_.workerId;}
    bool valid() const { return !outdated(job_);}
    bool validate() {
        if (valid()) {
            job_.t = std::chrono::steady_clock::now();
        }
        return valid();
    }
    void invalidate() {
        job_.t -= job_.ttl;
    }
    auto& ttl() const { return job_.ttl; }
    const LaunchId& launchId() const { return job_.launchId; }
};

class WorkerIdOutdated : public std::runtime_error {
public:
    using std::runtime_error::runtime_error;
};

template <typename WorkerId>
inline void throwIfOutdated(const WorkerId& id) {
    if (!id.valid()) {
        throw WorkerIdOutdated(id.id());
    }
}

} // namespace doberman

#endif /* DOBERMAN_SRC_META_TYPES_H_ */
