#ifndef DOBERMAN_SRC_PROFILING_PROFILER_H_
#define DOBERMAN_SRC_PROFILING_PROFILER_H_

#include <boost/any.hpp>
#include <chrono>
#include <src/detail/dereference.h>

namespace doberman {
namespace profiling {

using Milliseconds = std::chrono::milliseconds;
using Duration = Milliseconds;
using Time = std::chrono::time_point<std::chrono::system_clock, Milliseconds>;

inline Time now() {
    return std::chrono::time_point_cast<Milliseconds>(std::chrono::system_clock::now());
}

template <typename Impl>
class Profiler {
public:
    Profiler(Impl impl) : impl_(impl) {}

    void write(const std::string& caller,
               const std::string& action,
               Duration duration) const {
        impl().write(caller, action, duration);
    }

    void write(const std::string& caller,
               const std::string& action,
               const std::string& info,
               Duration duration) const {
        impl().write(caller, action, info, duration);
    }

    static Time now() {
        return std::chrono::time_point_cast<Milliseconds>(std::chrono::system_clock::now());
    }

    static auto passed(Time from) {
        return (now() - from);
    }

private:
    auto& impl() const { return ::doberman::detail::dereference(impl_); }
    Impl impl_;
};

template <typename Impl>
inline auto makeProfiler(Impl impl) {
    return Profiler<Impl>(std::move(impl));
}

} // namespace profiling

using profiling::Profiler;
using profiling::makeProfiler;

} // namespace doberman

#endif // DOBERMAN_SRC_PROFILING_PROFILER_H_
