#pragma once

#include <boost/asio/ip/tcp.hpp>
#include <boost/asio/spawn.hpp>
#include <boost/asio/streambuf.hpp>
#include <boost/asio/read_until.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include <src/service_control/command_type.h>
#include <src/service_control/run_status.h>
#include <src/logger/logger.h>

namespace doberman {
namespace service_control {

template <typename Log>
class CommandSourceImpl {
public:

    explicit CommandSourceImpl(boost::asio::io_service& service,
                               const RunStatus& runStatus,
                               Log& logger,
                               const boost::asio::ip::tcp::endpoint& endpoint)
        : acceptor(service, endpoint),
          socket(service),
          buffer(std::make_unique<boost::asio::streambuf>()),
          runStatus(runStatus),
          logger(logger)
    {}

    OptCommandType getCommand(boost::asio::yield_context context) {
        while (runStatus) {
            if (!socket.is_open()) {
                boost::system::error_code ec;
                acceptor.async_accept(socket, context[ec]);

                if (ec) {
                    DOBBY_LOG_ERROR(logger, "CommandSource: can't accept connection", log::error_code=ec);
                    return boost::none;
                }
            }

            const auto command = readCommand(context);
            if (command) {
                return command.get();
            }
        }

        closeSession();
        return boost::none;
    }

    void sendOkResponse(boost::asio::yield_context context) {
        sendResponse(context, "Ok\n");
    }

    void sendErrorResponse(boost::asio::yield_context context, const std::string& message) {
        const auto msg = boost::ends_with(message, "\n") ? message : message + '\n';
        sendResponse(context, msg);
    }

    void sendResponse(boost::asio::yield_context context, const std::string& response) {
        boost::system::error_code ec;
        socket.async_send(boost::asio::buffer(response), context[ec]);

        if (ec) {
            DOBBY_LOG_NOTICE(logger, "CommandSource: can't send response",
                    log::error_code=ec);
            closeSession();
        }
    }

private:

    using BufferPtr = std::unique_ptr<boost::asio::streambuf>;

    OptCommandType readCommand(boost::asio::yield_context context) {
        boost::system::error_code ec;
        boost::asio::async_read_until(socket, *buffer, '\n', context[ec]);

        if (ec) {
            closeSession();
            return boost::none;
        }

        std::istream stream(buffer.get());
        std::string commandString;
        std::getline(stream, commandString);

        return parseCommand(commandString);
    }

    void closeSession() {
        boost::system::error_code ec;
        socket.close(ec);
        if (ec) {
            DOBBY_LOG_ERROR(logger, "CommandSource: can't close socket", log::error_code=ec);
        }
        buffer->consume(buffer->size());
    }

    static CommandType parseCommand(const std::string& str) noexcept {
        if (str == "reopen_log") {
            return CommandType::reopenLog;
        } else if (str == "stop") {
            return CommandType::stop;
        } else if (str == "stat") {
            return CommandType::stat;
        } else {
            return CommandType::invalid;
        }
    }

    boost::asio::ip::tcp::acceptor acceptor;
    boost::asio::ip::tcp::socket socket;
    BufferPtr buffer;
    const RunStatus& runStatus;
    Log& logger;
};

template <typename Log>
auto makeCommandSourceImpl(boost::asio::io_service& service, const RunStatus& runStatus,
                           Log& logger, const boost::asio::ip::tcp::endpoint& endpoint) {
    return CommandSourceImpl<Log>(service, runStatus, logger, endpoint);
}

}}
