#pragma once

#include <boost/asio/io_service.hpp>
#include <boost/asio/spawn.hpp>

#include <src/detail/concept_check.h>
#include <src/service_control/command_type.h>
#include <src/service_control/run_status.h>
#include <src/logger/logger.h>

namespace doberman {
namespace service_control {

template <typename Log, typename LogSink, typename CmdSource>
class ServiceControl {
    using Context = boost::asio::yield_context;

    DOBBY_CHECK_CONCEPT(CmdSource,
        DOBBY_HAS_METHOD(getCommand, OptCommandType(Context))
        DOBBY_HAS_METHOD(sendOkResponse, void(Context))
        DOBBY_HAS_METHOD(sendErrorResponse, void(Context, const std::string&))
    )

    DOBBY_CHECK_CONCEPT(LogSink,
        DOBBY_HAS_METHOD(reopen, void())
    )
public:

    ServiceControl(boost::asio::io_service& service, Log& logger,
                   LogSink& logSink, CmdSource& cmdSource, RunStatus& runStatus,
                   std::function<std::string()> stat = []{ return "";})
        : logger(logger),
          logSink(logSink),
          cmdSource(cmdSource),
          service(service),
          runStatus(runStatus),
          stat_(stat)
    {}

    void run(boost::asio::yield_context context) {
        try {
            this->work(context);
        } catch (const std::exception& e) {
            DOBBY_LOG_ERROR(logger, "ServiceControl: internal error", log::exception=e);
            runStatus.reset();
        } catch (...) {
            DOBBY_LOG_ERROR(logger, "ServiceControl: unknown internal error");
            runStatus.reset();
        }
    }

private:

    void work(boost::asio::yield_context context) {
        while (runStatus) {
            const auto command = cmdSource.getCommand(context);
            if (command) {
                handleCommand(context ,command.get());
            }
        }
    }

    void handleCommand(boost::asio::yield_context context, const CommandType commandType) {
        switch (commandType) {
        case CommandType::reopenLog:
            rotateLog(context);
            break;
        case CommandType::stop:
            runStatus.reset();
            cmdSource.sendOkResponse(context);
            break;
        case CommandType::stat:
            cmdSource.sendResponse(context, stat_());
            break;
        case CommandType::invalid:
            cmdSource.sendErrorResponse(context, "Invalid command\n");
            break;
        }
    }

    void rotateLog(boost::asio::yield_context context) {
        try {
            logSink.reopen();
            cmdSource.sendOkResponse(context);
        } catch (const std::exception& e) {
            DOBBY_LOG_ERROR(logger, "ServiceControl: reopen log failed", log::exception=e);
            cmdSource.sendErrorResponse(context, "Reopen log failed");
        }
    }

    Log& logger;
    LogSink& logSink;
    CmdSource& cmdSource;
    boost::asio::io_service& service;
    RunStatus& runStatus;
    std::function<std::string()> stat_;
};

template <typename Log, typename LogSink, typename CmdSource>
auto makeServiceControl(boost::asio::io_service& service, Log& log,
                        LogSink& logSink, CmdSource& cmdSource, RunStatus& runStatus,
                        std::function<std::string()> stat = []{ return "";}) {
    return ServiceControl<Log, LogSink, CmdSource>(service, log, logSink, cmdSource, runStatus, stat);
}

}}
