#!/usr/bin/env python3

import click


class Context(dict):
    def __getitem__(self, key):
        return super().get(key) or f'${key}'


def wait_for_enter():
    input("Press Enter to continue: ")
    print()


class Step:
    def run(self, context):
        pass


class CreateShard(Step):
    def run(self, context):
        print("* Prerequisites:")
        print("* Installed `yc` CLI")
        print("* Installed `pgmigrate` utility")
        print("* Checked-out Arcadia with `mail/pg/mdb` and `mail/tools/dbaas` directories")

        env = context['env']
        arcadia_root = context['arcadia_root']
        migrations_dir = context.get('migrations_dir') or f'{arcadia_root}/mail/pg/mdb'
        print("Go to https://yav.yandex-team.ru/secret/sec-01d5v5dv67pza174zneyt5s1tf/explore/versions and download users.yaml file for your env into local file in CWD")
        wait_for_enter()
        print("To create shard in MDB, run:")
        print(f"python3 {arcadia_root}/mail/tools/dbaas/create_shard.py --env {env} --migrations-dir {migrations_dir}")
        context['shard_name'] = input('Type resulting shard name (string between `maildb_` and `_pgcluster`): ')
        print()


class EnsureMasterNotMan(Step):
    def run(self, context):
        print("Go to https://yc.yandex.team.ru , find created cluster and check that master is not in `man` DC.")
        print("If it is in `man`, force it to become a replica by enabling and then disabling 'Replication source' for host in `man`.")
        print("Or even better, ask in Telegram chat `mdb-support` for master switch.")
        wait_for_enter()


class AddToSharpei(Step):
    def run(self, context):
        shard_name = context['shard_name']
        print("Connect to sharddb master via `psql 'host=c-mdbd8ebfpfh8ult1t721.rw.db.yandex.net port=6432 dbname=sharddb user=sharpei'` and run following queries:")
        print(f"select code.add_cluster('{shard_name}', 'dbaas_hot', array[('man-abc123', 6432, 'man', 'maildb'), (...), (...)]::code.instance[]);")
        print()
        print("===After this step, real workload is gonna go on shard===")
        print("To enable registration on this shard, use this query:")
        print(f"update shards.shards set reg_weight = 100 where shard_id = {context['shard_id']};")
        print()
        print("To enable transfer on this shard, use this query:")
        print(f"insert into shards.shards_load values ({context['shard_id']}, 0, true);")
        wait_for_enter()


class RegDobermanJobs(Step):
    def run(self, context):
        env = context['env']
        shard_name = context['shard_name']
        if env == 'corp':
            print(f"Connect to shard via `ya-xdb -e {env} {shard_name}` and run following command:")
            print("insert into mail.doberman_jobs(worker_id) values ('husky'), ('dobby');")


class AddToSamsara(Step):
    def run(self, context):
        shard_name = context['shard_name']
        print("Go to https://platform.yandex-team.ru/projects/mail/samsara/production => Create Draft")
        print(f"Copy any of existing component, name it `samsara-{context['shard_id']}`, enter Edit")
        print(f"Go to `Variables` tab, edit CRON_SHARD_ID={context['shard_id']}")
        print(f"Save results, go `Deploy Draft`, confirm changes by pressing `Deploy`")
        wait_for_enter()


class WatchGraphics(Step):
    def run(self, context):
        shard_name = context['shard_name']
        print(f"Go to https://yasm.yandex-team.ru/template/panel/Maildb-dbstats/ctype={shard_name}/ and check if graphics are there and everything is okay")
        wait_for_enter()


@click.command()
@click.option('--env')
@click.option('--migrations-dir')
@click.option('--arcadia-root')
def main(**kwargs):
    context = Context(**kwargs)
    procedure = [
        CreateShard(),
        EnsureMasterNotMan(),
        AddToSharpei(),
        RegDobermanJobs(),
        AddToSamsara(),
        WatchGraphics(),
    ]
    for step in procedure:
        step.run(context)
    print("Done.")


if __name__ == '__main__':
    main()
