#!/usr/bin/env python3

import click


class Context(dict):
    def __getitem__(self, key):
        return super().get(key) or f'${key}'


def wait_for_enter():
    input("Press Enter to continue: ")
    print()


class Step:
    def run(self, context):
        pass


class MigrateShards(Step):
    def run(self, context):
        print("* Prerequisites:")
        print("* Installed `yc` CLI")
        print("* Installed `pgmigrate` utility")
        print("* Checked-out Arcadia with `mail/pg/mdb` and `mail/tools/dbaas` directories")
        print()

        arcadia_root = context['arcadia_root']
        migration_tool_dir = f'{arcadia_root}/mail/tools/dbaas/bin/migrate_shards'
        migration_tool = f'{migration_tool_dir}/migrate_shards'
        migrations_dir = context.get('migrations_dir') or f'{arcadia_root}/mail/pg/mdb'
        print("Build migration tool by running following command:")
        print(f"{arcadia_root}/ya make --checkout {migration_tool_dir}")
        wait_for_enter()
        print("Check out the migration tool help:")
        print(f"{migration_tool} --help")
        wait_for_enter()
        print("Migrate load shards in MDB by running following command:")
        print(f"{migration_tool} --env load --migrations-dir {migrations_dir}")
        print("Add --update-cluster flag if you need to create new role, create new extension or update cluster settings")
        wait_for_enter()
        print("Migrate testing shards in MDB by running following command:")
        print(f"{migration_tool} --env test --migrations-dir {migrations_dir}")
        print("Add --update-cluster flag if you need to create new role, create new extension or update cluster settings")
        wait_for_enter()
        print("Then, migrate intranet-production shards in MDB by running following command:")
        print(f"{migration_tool} --env corp --migrations-dir {migrations_dir}")
        print("Add --update-cluster flag if you need to create new role, create new extension or update cluster settings")
        wait_for_enter()
        print("Then, migrate production shards in MDB by running following command:")
        print(f"{migration_tool} --env prod --migrations-dir {migrations_dir}")
        print("Add --update-cluster flag if you need to create new role, create new extension or update cluster settings")
        wait_for_enter()
        print()


@click.command()
@click.option('--migrations-dir')
@click.option('--arcadia-root')
def main(**kwargs):
    context = Context(**kwargs)
    procedure = [
        MigrateShards(),
    ]
    for step in procedure:
        step.run(context)
    print("Done.")


if __name__ == '__main__':
    main()
