#pragma once

#include <variant>
#include <optional>

#include <boost/system/error_code.hpp>
#include <boost/system/system_error.hpp>

#include <mail_errors/error_code.h>

namespace yamail {

template <class E>
class bad_expected_access : public std::logic_error {
    E e_;
public:
    using error_type = E;
    explicit bad_expected_access(error_type e)
    : std::logic_error("error while access value expected"), e_(std::move(e)){}
    constexpr error_type const& error() const { return e_;}
};

template <>
class bad_expected_access<mail_errors::error_code> : public std::logic_error {
    mail_errors::error_code e_;
public:
    using error_type = mail_errors::error_code;
    explicit bad_expected_access(error_type e)
    : std::logic_error(e.what()), e_(std::move(e)) {}
    constexpr error_type const& error() const { return e_;}
};

template <class E>
class unexpected_type {
public:
    using error_type = E;
    explicit unexpected_type(error_type const& v) : val(v) {}
    explicit unexpected_type(error_type&& v) : val(v) {}
    error_type& value() & noexcept { return val;}
    error_type&& value() && noexcept { return std::move(val);}
    error_type const & value() const& noexcept { return val;};
private:
    E val;
};

template <typename T>
struct is_expected : std::false_type {};

template <typename T>
constexpr auto is_expected_v = is_expected<T>::value;

template <typename F, typename ...Ts>
constexpr auto returns_void_v = std::is_same_v<std::invoke_result_t<F, Ts...>, void>;

template <typename F, typename ...Ts>
constexpr auto returns_expected_v = is_expected_v<std::invoke_result_t<F, Ts...>>;

/**
 * @brief expected monad proposal like class
 *
 * This class models expected monad in a basic way to eliminate copypasted
 * reinvented wheels and make easer to move to standard implementation later.
 */
template <typename T, typename E = mail_errors::error_code>
class expected {
public:
    using value_type = T;
    using error_type = E;

    template <class U>
    struct rebind { using type = expected<U, error_type>; };
    template <class U>
    using rebind_t = typename rebind<U>::type;
    template <typename F>
    using rebind_func_t = rebind_t<std::invoke_result_t<F, value_type>>;

    expected() = default;
    expected(expected&&) = default;
    expected(const expected&) = default;

    expected(value_type&& v) : v_(std::move(v)) {}
    expected(const value_type& v) : v_(v) {}

    expected(unexpected_type<error_type>&& v) : v_(std::in_place_type<error_type>, std::move(v.value())) {}
    expected(const unexpected_type<error_type>& v) : v_(std::in_place_type<error_type>, v.value()) {}

    expected& operator=(expected&&) = default;
    expected& operator=(const expected&) = default;

    T const* operator ->() const { return std::addressof(value()); }
    T* operator ->() { return std::addressof(value()); }

    T const& operator *() const& { return value(); }
    T& operator *() & { return value(); }
    T&& operator *() &&  { return value(); }

    explicit operator bool() const noexcept { return has_value();}

    T const& value() const& {
        throw_if_unexpected();
        return std::get<value_type>(v_);
    }
    T& value() & {
        throw_if_unexpected();
        return std::get<value_type>(v_);
    }
    T&& value() && {
        throw_if_unexpected();
        return std::get<value_type>(std::move(v_));
    }

    auto get_unexpected() const {
        return unexpected_type<error_type>(error());
    }

    E const& error() const& { return std::get<error_type>(v_);}
    E& error() & { return std::get<error_type>(v_);}
    E&& error() && { return std::get<error_type>(std::move(v_));}

    template <typename F>
    auto map(F&& func) -> std::enable_if_t<
            !returns_void_v<F, value_type>,
            rebind_func_t<F>> {
        if (*this) {
            return func(std::move(value()));
        }
        return get_unexpected();
    }

    template <typename F>
    auto map(F&& func) -> std::enable_if_t<
            returns_void_v<F, value_type>,
            rebind_t<void>> {
        if (*this) {
            func(std::move(value()));
            return {};
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            returns_expected_v<F, value_type>,
            std::invoke_result_t<F, value_type>> {
        if (*this) {
            return func(std::move(value()));
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            !(returns_expected_v<F, value_type> || returns_void_v<F, value_type>),
            rebind_func_t<F>> {
        if (*this) {
            return func(std::move(value()));
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            returns_void_v<F, value_type>,
            rebind_t<void>> {
        if (*this) {
            func(std::move(value()));
            return {};
        }
        return get_unexpected();
    }
    ///

    template <typename F>
    auto catch_error(F&& f) const -> std::enable_if_t<
            std::is_same_v<std::invoke_result_t<F, error_type>, value_type>
            || std::is_same_v<std::invoke_result_t<F, error_type>, expected>
            || std::is_same_v<std::invoke_result_t<F, error_type>, unexpected_type<error_type>>,
            expected> {
        if(!*this) {
            return f(error());
        }
        return *this;
    }

    template <typename F>
    auto catch_error(F&& f) const -> std::enable_if_t<
            returns_void_v<F, error_type>,
            expected> {
        if(!*this) {
            f(error());
            return {};
        }
        return *this;
    }

    const T& value_or_throw() const & {
        if(!*this) {
            throw mail_errors::system_error(error());
        }
        return std::get<value_type>(v_);
    }

    T& value_or_throw() & {
        if(!*this) {
            throw mail_errors::system_error(error());
        }
        return std::get<value_type>(v_);
    }

    T&& value_or_throw() && {
        if(!*this) {
            throw mail_errors::system_error(error());
        }
        return std::get<value_type>(std::move(v_));
    }

    template <typename F>
    auto then(F&& func) -> rebind_t<decltype(func(*this))> {
        return func(std::move(*this));
    }

private:
    bool has_value() const noexcept {
        return !std::holds_alternative<error_type>(v_);
    }

    void throw_if_unexpected() const {
        if (!has_value()) {
            throw bad_expected_access(error());
        }
    }

    std::variant<value_type, error_type> v_;
};

template <class T, class E>
constexpr bool operator==(const expected<T,E>& lhs, const expected<T,E>& rhs) {
  return (lhs && rhs) ? *lhs == *rhs :
        (!lhs && !rhs) ?  lhs.error() == rhs.error() : false;
}

template <class E>
constexpr bool operator==(const expected<void, E>& lhs, const expected<void, E>& rhs) {
  return (lhs && rhs) || (!lhs && !rhs && lhs.error() == rhs.error());
}

template <class T, class E>
constexpr bool operator!=(const expected<T,E>& lhs, const expected<T,E>& rhs) {
    return !(lhs == rhs);
}

template <class T, class E>
constexpr bool operator==(const expected<T,E>& lhs, const T& rhs) {
    return lhs ? *lhs == rhs : false;
}
template <class T, class E>
constexpr bool operator==(const T& lhs, const expected<T,E>& rhs) {
    return rhs == lhs;
}
template <class T, class E>
constexpr bool operator!=(const expected<T,E>& lhs, const T& rhs) {
    return !(lhs == rhs);
}
template <class T, class E>
constexpr bool operator!=(const T& lhs, const expected<T,E>& rhs) {
    return !(lhs == rhs);
}

template <class T, class E>
constexpr bool operator==(const expected<T,E>& lhs, const unexpected_type<E>& rhs) {
    return lhs ? false : lhs.error() == rhs.value();
}
template <class T, class E>
constexpr bool operator==(const unexpected_type<E>& lhs, const expected<T,E>& rhs) {
    return rhs == lhs;
}
template <class T, class E>
constexpr bool operator!=(const expected<T,E>& lhs, const unexpected_type<E>& rhs) {
    return !(lhs == rhs);
}
template <class T, class E>
constexpr bool operator!=(const unexpected_type<E>& lhs, const expected<T,E>& rhs) {
    return !(lhs == rhs);
}

template <typename E>
class expected<void, E> {
public:
    using value_type = void;
    using error_type = E;

    template <class U>
    struct rebind { using type = expected<U, error_type>; };
    template <class U>
    using rebind_t = typename rebind<U>::type;
    template <typename F>
    using rebind_func_t = rebind_t<std::invoke_result_t<F>>;

    expected() = default;
    expected(expected&&) = default;
    expected(const expected&) = default;
    expected(unexpected_type<error_type>&& v) : v_(std::move(v.value())) {}
    expected(const unexpected_type<error_type>& v) : v_(v.value()) {}

    expected& operator=(expected&&) = default;
    expected& operator=(const expected&) = default;

    explicit operator bool() const noexcept { return has_value(); }

    void value() const { throw_if_unexpected(); }

    auto get_unexpected() const {
        return unexpected_type<error_type>(error());
    }

    E const& error() const& { return std::get<error_type>(v_);}
    E& error() & { return std::get<error_type>(v_);}
    E&& error() && { return std::get<error_type>(std::move(v_));}

    template <typename F>
    auto map(F&& func) -> std::enable_if_t<
            !returns_void_v<F>,
            rebind_func_t<F>> {
        if (*this) {
            return func();
        }
        return get_unexpected();
    }

    template <typename F>
    auto map(F&& func) -> std::enable_if_t<
            returns_void_v<F>,
            expected> {
        if (*this) {
            func();
            return {};
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            returns_expected_v<F>,
            std::invoke_result_t<F>> {
        if (*this) {
            return func();
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            !(returns_expected_v<F> || returns_void_v<F>),
            rebind_func_t<F>> {
        if (*this) {
            return func();
        }
        return get_unexpected();
    }

    template <typename F>
    auto bind(F&& func) -> std::enable_if_t<
            returns_void_v<F>,
            expected> {
        if (*this) {
            func();
            return {};
        }
        return get_unexpected();
    }

    template <typename F>
    auto catch_error(F&& f) -> std::enable_if_t<
            std::is_same_v<std::invoke_result_t<F, error_type>, expected>
            || std::is_same_v<std::invoke_result_t<F, error_type>, unexpected_type<error_type>>,
            expected> {
        if(!*this) {
            return f(error());
        }
        return *this;
    }

    template <typename F>
    auto catch_error(F&& f) const -> std::enable_if_t<
            returns_void_v<F, error_type>,
            expected> {
        if(!*this) {
            f(error());
            return {};
        }
        return *this;
    }

    void value_or_throw() const {
        if(!*this) {
            throw mail_errors::system_error(error());
        }
    }

    template <typename F>
    auto then(F&& func) -> rebind_t<decltype(func(*this))> {
        return func(std::move(*this));
    }
private:
    bool has_value() const noexcept {
        return !std::holds_alternative<error_type>(v_);
    }
    void throw_if_unexpected() const {
        if (!has_value()) {
            throw bad_expected_access(error());
        }
    }
    std::variant<std::monostate, error_type> v_;
};

template <typename T, typename E>
struct is_expected<expected<T, E>> : std::true_type {};

template <typename T>
inline auto make_expected(T&& v) {
    using result_type = expected<std::decay_t<T>, mail_errors::error_code>;
    return result_type{std::forward<T>(v)};
}

inline auto make_expected() {
    return expected<void, mail_errors::error_code>{};
}

template <typename E>
inline constexpr auto make_unexpected(E&& e) {
    return unexpected_type<std::decay_t<E>>{std::forward<E>(e)};
}

inline auto make_unexpected(mail_errors::error_code e) {
    return unexpected_type<mail_errors::error_code>{std::move(e)};
}

template <class T, class E>
inline auto make_expected_from_error(E&& e) {
    return expected<T, std::decay_t<E>>{make_unexpected(std::forward<E>(e))};
}

} // namespace yamail
