import logging
from data_logging.formatters import (
    json,
    JSONFormatter as JSONFormatterBase,
    DEFAULT_ORDERED_FIRST,
    DEFAULT_ORDERED_LAST,
    DEFAULT_RENAME_FIELDS,
    DEFAULT_EXCLUDED_FIELDS,
    RecordDataFormatterMixin,
)
from data_logging.subsix import text
from . import utils


__all__ = (
    "JSONFormatter",
    "TSKVFormatterBase",
    "TSKVFormatter",
)


class DataOverrideKeyMixin:
    """
    A helper whose primary use is logging data with 'name', 'exc_info' and so
    on on the upper level through the `RecordDataFormatterMixin`.
    """

    data_override_key = "_data_override"

    def record_to_data(self, record, **kwargs):
        data = super().record_to_data(record, **kwargs)
        data_overrides, data = utils.split_list(
            data, lambda item: item[0] == self.data_override_key
        )
        for data_override_item in data_overrides:
            _, data_override = data_override_item
            data_override = data_override.copy()
            data = [(key, data_override.pop(key, value)) for key, value in data]
            data.extend(list(data_override.items()))
        return data


class JSONFormatter(DataOverrideKeyMixin, JSONFormatterBase):
    """..."""


class QloudJSONFormatter(JSONFormatter):

    qloud_levels = set(
        (
            "ERROR",
            "ERR",
            "error",
            "err",
            "WARNING",
            "WARN",
            "warning",
            "warn",
            "INFO",
            "info",
            "DEBUG",
            "debug",
        )
    )

    def _serialize(self, obj):
        """
        Mangle the object to conform to qloud:
        https://docs.qloud.yandex-team.ru/doc/logs#json
        """
        data = {"@fields": obj}
        data["message"] = obj.get("message")
        level = obj.get("levelname")
        data["level"] = level if level in self.qloud_levels else "info"
        exc_info = obj.get("exc_info")
        if exc_info is not None and isinstance(exc_info, (bytes, text)):
            data["stackTrace"] = exc_info
        return super()._serialize(data)


class TSKVFormatterBase(RecordDataFormatterMixin, logging.Formatter):

    encode_default = None

    def __init__(self, tskv_format, *args, **kwargs):
        """
        A logging formatter that turns a record into TSKV, for convenient automatic processing.

        :param tskv_format: value for the `tskv_format` field, which
        is used to determine the target table on the cluster.

        """

        encode_default = kwargs.pop("encode_default", "json")
        if encode_default == "str":
            encode_default = str
        elif encode_default == "json":
            encode_default = self._json_encode
        assert not isinstance(encode_default, (bytes, text))
        self.encode_default = encode_default
        # TODO?: max_message_length, truncate_warning_text
        self.tskv_format = tskv_format
        super().__init__(*args, **kwargs)

    __init__.__doc__ += RecordDataFormatterMixin.__init__.__doc__  # pylint: disable=no-member

    def list_autogenerated_fields(self):
        return ("tskv_format",) + super().list_autogenerated_fields()

    def fields_check(self, fields):
        assert "tskv_format" in fields, "tksv_format must be included"
        assert ("timestamp" in fields and "timezone" in fields) or (
            "unixtime" in fields
        ), "either timestamp+timezone or unixtime must be included"
        return fields

    def make_record_field_overrides(self):
        return dict(
            super().make_record_field_overrides(),
            tskv_format=lambda record: self.tskv_format,
        )

    @staticmethod
    def _json_encode(obj):
        try:
            result = json.dumps(obj, ensure_ascii=False, default=repr)
            if isinstance(result, text):
                result = result.encode("utf-8", errors="replace")
        except Exception:
            return b"<error dumping the value>"
        return result

    def tskv_encode(self, data):
        # Extra newline is useful for e.g. `tail -f` on logs.
        return utils.tskv_encode(data, default=self.encode_default) + b"\n"

    def format(self, record):
        data = self.record_to_data(record)
        return self.tskv_encode(data)


class TSKVFormatterStrict(TSKVFormatterBase):
    """
    An example TSKV formatter that keeps an explicit list of
    fields on the upper level and serializes everything else into JSON
    on an additional field.
    """

    _extras_field = "extras"
    _upper_fields = set()

    def format(self, record):
        data = self.record_to_data(record)
        upper_fields = self._upper_fields

        data_upper, data_lower = utils.split_list(data, (lambda name, val: name in upper_fields))
        data = data_upper + [(self._extras_field, json.dumps(data_lower))]
        return self.tskv_encode(data)


class TSKVFormatter(TSKVFormatterBase):
    """TSKVFormatterBase + useful defaults"""

    def __init__(self, *args, **kwargs):
        kwargs.setdefault("exclude_fields", DEFAULT_EXCLUDED_FIELDS)
        kwargs.setdefault("ordered_first", DEFAULT_ORDERED_FIRST)
        kwargs.setdefault("ordered_last", DEFAULT_ORDERED_LAST)
        kwargs.setdefault("rename_fields", DEFAULT_RENAME_FIELDS)
        kwargs.setdefault("sort_other_fields", True)
        super().__init__(*args, **kwargs)
