#pragma once
// DO_NOT_STYLE

#include <stddef.h>
#include <stdint.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

enum ETicketParser2ErrorCode {
    TP2EC_OK = 0,
    TP2EC_DEPRECATED,
    TP2EC_EMPTY_TVM_KEYS,
    TP2EC_EXPIRED_TICKET,
    TP2EC_INVALID_BLACKBOX_ENV,
    TP2EC_INVALID_DST,
    TP2EC_INVALID_PARAM,
    TP2EC_INVALID_TICKET_TYPE,
    TP2EC_MALFORMED_TICKET,
    TP2EC_MALFORMED_TVM_KEYS,
    TP2EC_MALFORMED_TVM_SECRET,
    TP2EC_MISSING_KEY,
    TP2EC_NOT_ALLOWED,
    TP2EC_SIGN_BROKEN,
    TP2EC_SMALL_BUFFER,
    TP2EC_UNEXPECTED_ERROR,
    TP2EC_UNSUPPORTED_VERSION,
};
struct TTP2_ServiceContext;
struct TTP2_ServiceTicket;
struct TTP2_UserContext;
struct TTP2_UserTicket;

extern const char* TP2_BlackboxClientIdProd;       /* 222 */
extern const char* TP2_BlackboxClientIdTest;       /* 224 */
extern const char* TP2_BlackboxClientIdProdYateam; /* 223 */
extern const char* TP2_BlackboxClientIdTestYateam; /* 225 */
extern const char* TP2_BlackboxClientIdStress;     /* 226 */
extern const char* TP2_BlackboxClientIdMimino;     /* 239 */

enum TP2_EBlackboxEnv {
    EBE_PROD = 0,
    EBE_TEST,
    EBE_PROD_YATEAM,
    EBE_TEST_YATEAM,
    EBE_STRESS
};

const char* TP2_ErrorCodeToString(enum ETicketParser2ErrorCode code);

/*!
 * Create service context. Serivce contexts are used to store TVM keys and parse service tickets.
 * @param[in] clientId
 * @param[in] secretBase64 Secret, can be NULL. Sign attempt hence raises exception
 * @param[in] secretBase64Size Size of string containing secret
 * @param[in] tvmKeysResponse TVM Keys gotten from TVM API
 * @param[in] tvmKeysResponseSize Size of string containing keys
 * @param[out] context
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_CreateServiceContext(
    uint32_t clientId,
    const char* secretBase64,
    size_t secretBase64Size,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize,
    struct TTP2_ServiceContext** context);

/*!
 * Free memory of service context.
 * @param[in] context
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_DeleteServiceContext(struct TTP2_ServiceContext* context);

/*!
 * Parse and validate service ticket body then create TServiceTicket object.
 * @param[in] context
 * @param[in] ticketBody Service ticket body as string
 * @param[in] ticketBodySize Size of string containing service ticket body
 * @param[out] ticket Service ticket object
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_CheckServiceTicket(
    const struct TTP2_ServiceContext* context,
    const char* ticketBody,
    size_t ticketBodySize,
    struct TTP2_ServiceTicket** ticket);

/*!
 * Free memory of service ticket.
 * @param[in] ticket
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_DeleteServiceTicket(struct TTP2_ServiceTicket* ticket);

/*!
 * Source of request, your client.
 * @param[in] ticket Parsed ticket
 * @param[out] srcClientId Integer identifier of client
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetServiceTicketSrc(
    const struct TTP2_ServiceTicket* ticket,
    uint32_t* srcClientId);

/*!
 * Return number of scopes in the ticket.
 * @param[in] ticket
 * @param[out] count
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetServiceTicketScopesCount(
    const struct TTP2_ServiceTicket* ticket,
    size_t* count);

/*!
 * Return scope by serial index.
 * WARNING: Do not free returned pointer.
 * @param[in] ticket
 * @param[in] idx
 * @param[out] scope
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetServiceTicketScope(
    const struct TTP2_ServiceTicket* ticket,
    size_t idx,
    const char** scope);

/*!
 * Check if service ticket has the scope
 * @param[in] ticket
 * @param[in] scope
 * @param[out] checkingResult Equal to 1 if scope is present in ticket and to 0 otherwise
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_HasServiceTicketScope(
    const struct TTP2_ServiceTicket* ticket,
    const char* scope,
    size_t scopeSize,
    int* checkingResult);

/*!
 * Return debug info for ticket
 * @param[in] ticket
 * @param[out] debugInfo
 * @param[out] debugInfoSize
 * @return Error code
 * @param[in] maxDebugInfoSize
 */
enum ETicketParser2ErrorCode TP2_GetServiceTicketDebugInfo(
    const struct TTP2_ServiceTicket* ticket,
    char* debugInfo,
    size_t* debugInfoSize,
    size_t maxDebugInfoSize);

/*!
 * Return part of ticket which can be safely logged.
 * WARNING: Do not free returned pointer.
 * WARNING: logableTicket is not zero-ended string (it is substring of ticketBody)
 * WARNING: Use logableTicketSize to read valid amount of symbols.
 * @param[in] ticketBody
 * @param[in] ticketBodySize
 * @param[out] logableTicket
 * @param[out] logableTicketSize
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_RemoveServiceTicketSignature(
    const char* ticketBody,
    size_t ticketBodySize,
    const char** logableTicket,
    size_t* logableTicketSize);

/*!
 * Set new keys for existing service context.
 * WARNING: Thread unsafe
 * @param[in] context
 * @param[in] tvmKeysResponse TVM Keys gotten from TVM API
 * @param[in] tvmKeysResponseSize Size of string containing keys
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_ResetServiceContextKeys(
    struct TTP2_ServiceContext* context,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize);

/*!
 * Create signature for selected parameters. Allocate at least 512 byte for signature buffer.
 * @param[in] context
 * @param[in] ts Param 'ts' of request to TVM
 * @param[in] tsSize Size of param 'ts' of request to TVM
 * @param[in] dst Param 'dst' of request to TVM
 * @param[in] dstSize Size of param 'dst' of request to TVM
 * @param[in] scopes Param 'scopes' of request to TVM
 * @param[in] scopesSize Size of param 'scopes' of request to TVM
 * @param[out] signature
 * @param[out] signatureSize
 * @param[in] maxSignatureSize
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_SignCgiParamsForTvm(
    const struct TTP2_ServiceContext* context,
    const char* ts,
    size_t tsSize,
    const char* dst,
    size_t dstSize,
    const char* scopes,
    size_t scopesSize,
    char* signature,
    size_t* signatureSize,
    size_t maxSignatureSize);

/*!
 * Create user context. User contexts are used to store TVM keys and parse user tickets.
 * @param[in] env
 * @param[in] tvmKeysResponse
 * @param[in] tvmKeysResponseSize
 * @param[out] context
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_CreateUserContext(
    enum TP2_EBlackboxEnv env,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize,
    struct TTP2_UserContext** context);

/*!
 * Free memory of user context.
 * @param[in] context
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_DeleteUserContext(struct TTP2_UserContext* context);

/*!
 * Parse and validate user ticket body then create TUserTicket object.
 * @param[in] context
 * @param[in] ticketBody Service ticket body as string
 * @param[in] ticketBodySize Size of string containing service ticket body
 * @param[out] ticket Service ticket object
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_CheckUserTicket(
    const struct TTP2_UserContext* context,
    const char* ticketBody,
    size_t ticketBodySize,
    struct TTP2_UserTicket** ticket);

/*!
 * Free memory of user ticket.
 * @param[in] context
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_DeleteUserTicket(struct TTP2_UserTicket* context);

/*!
 * Return number of UIDs in the ticket.
 * @param[in] ticket
 * @param[out] count
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketUidsCount(
    const struct TTP2_UserTicket* ticket,
    size_t* count);

/*!
 * Return UID from ticket by ordinal index
 * @param[in] ticket
 * @param[in] idx
 * @param[out] uid
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketUid(
    const struct TTP2_UserTicket* ticket,
    size_t idx,
    uint64_t* uid);

/*!
 * Return default UID. Default UID is the chosen one which should be considered as primary.
 * @param[in] ticket
 * @param[out] uid
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketDefaultUid(
    const struct TTP2_UserTicket* ticket,
    uint64_t* uid);

/*!
 * Return number of scopes in the ticket.
 * @param[in] ticket
 * @param[out] count
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketScopesCount(
    const struct TTP2_UserTicket* ticket,
    size_t* count);

/*!
 * Return scope by ordinal index.
 * WARNING: Do not free returned pointer.
 * @param[in] ticket
 * @param[in] idx
 * @param[out] scope
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketScope(
    const struct TTP2_UserTicket* ticket,
    size_t idx,
    const char** scope);

/*!
 * Check if user ticket has the scope
 * @param[in] ticket
 * @param[in] scope
 * @param[in] scopeSize Size of string containing scope
 * @param[out] checkingResult Equal to 1 if scope is present in ticket and to 0 otherwise
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_HasUserTicketScope(
    const struct TTP2_UserTicket* ticket,
    const char* scope,
    size_t scopeSize,
    int* checkingResult);

/*!
 * Return debug info for ticket
 * @param[in] ticket
 * @param[out] debugInfo
 * @param[out] debugInfoSize
 * @param[in] maxDebugInfoSize
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_GetUserTicketDebugInfo(
    const struct TTP2_UserTicket* ticket,
    char* debugInfo,
    size_t* debugInfoSize,
    size_t maxDebugInfoSize);

/*!
 * Return part of ticket which can be safely logged.
 * WARNING: Do not free returned pointer.
 * WARNING: logableTicket is not zero-ended string (it is substring of ticketBody)
 * WARNING: Use logableTicketSize to read valid amount of symbols.
 * @param[in] ticketBody
 * @param[in] ticketBodySize
 * @param[out] logableTicket
 * @param[out] logableTicketSize
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_RemoveUserTicketSignature(
    const char* ticketBody,
    size_t ticketBodySize,
    const char** logableTicket,
    size_t* logableTicketSize);

/*!
 * Set new keys for existing user context.
 * WARNING: Thread unsafe
 * @param[in] context
 * @param[in] tvmKeysResponse
 * @param[in] tvmKeysResponseSize
 * @return Error code
 */
enum ETicketParser2ErrorCode TP2_ResetUserContextKeys(
    struct TTP2_UserContext* context,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize);

/*!
 * Return library version.
 * @return version
 */
const char* TP2_LibVersion();

#ifdef __cplusplus
}
#endif
