from datetime import datetime


class BaseRecord:
    def _make_tskv_value(self, name, value):
        if value is None or value == "":
            return None
        return "{}={}".format(name, value)

    def _to_tskv(self):
        if not self._valid:
            raise ValueError("record is incorrect")
        parts = ("tskv\tdate={}".format(self.date),)
        return "\t".join((x for x in parts if x))

    @property
    def valid(self):
        return self._valid

    @property
    def tskv(self):
        return self._to_tskv() + "\n"


class LogRecord(BaseRecord):
    def __init__(
        self,
        timestamp=None,
        event=None,
        email=None,
        campaign=None,
        account=None,
        user_agent=None,
        user_ip=None,
        test="False",
        **kwargs
    ):
        self._valid = True
        if not timestamp or not campaign or not email or test == "True":
            self._valid = False
            return
        try:
            event_date = datetime.strptime(timestamp, "%Y-%m-%d %H:%M:%S.%f")
            self.date = event_date.strftime("%Y-%m-%d")
        except ValueError:
            self._valid = False
            return
        self.timestamp = timestamp
        self.recipient = email
        self.event = event
        self.event_date = event_date.strftime("%Y-%m-%d %H:%M:%S")
        self.campaign = int(campaign) if campaign and campaign.isdigit() else None
        self.account = account
        self.user_agent = user_agent
        self.user_ip = user_ip

    def _to_tskv(self):
        parts = (
            super()._to_tskv(),
            self._make_tskv_value("recipient", self.recipient),
            self._make_tskv_value("event", self.event),
            self._make_tskv_value("campaign", self.campaign),
            self._make_tskv_value("account", self.account),
            self._make_tskv_value("user_agent", self.user_agent),
            self._make_tskv_value("user_ip", self.user_ip),
        )
        return "\t".join((x for x in parts if x))
