import logging
import random
from collections import deque

from django.conf import settings

import requests

from .. import TSKV
from .parsers import LogParser

loger = logging.getLogger(__name__)


class ClickhouseTSKVPusher:
    def __init__(self, logger, table_name):
        self.logger = logger
        self.__query = "INSERT INTO {}.{} FORMAT TSKV".format(
            settings.CLICKHOUSE_DATABASE, table_name
        )
        self.__timeout = 5
        self.__url = self._make_url_with_random_host()

    def push(self, records):
        if not records:
            return

        tskv_gen = (rec.tskv for rec in records)
        push_data = "".join(tskv_gen).encode("utf-8")

        while True:  # Cross host retries
            try:
                r = requests.post(
                    self.__url,
                    params={"query": self.__query},
                    data=push_data,
                    timeout=self.__timeout,
                )
                if r.status_code == requests.codes.ok:
                    break
                else:
                    self.logger.error(r.text)
            except requests.exceptions.RequestException as exc:
                self.logger.exception(exc)
                self.__url = self._make_url_with_random_host()

        self.logger.debug("data pushed to clickhouse")

    def _make_url_with_random_host(self):
        # workaround for local env
        if settings.CLICKHOUSE_USERNAME is None:
            return "{scheme}://{host}:{port}/".format(
                scheme="http" if settings.CLICKHOUSE_PORT == 8123 else "https",
                host=random.choice(settings.CLICKHOUSE_HOSTS),
                port=settings.CLICKHOUSE_PORT,
            )
        else:
            return "{scheme}://{user}:{password}@{host}:{port}/".format(
                scheme="http" if settings.CLICKHOUSE_PORT == 8123 else "https",
                user=settings.CLICKHOUSE_USERNAME,
                password=settings.CLICKHOUSE_PASSWORD,
                host=random.choice(settings.CLICKHOUSE_HOSTS),
                port=settings.CLICKHOUSE_PORT,
            )


class BaseProcessor:
    """Базовый класс процессоров статистики Рассылятора"""

    _parser_cls = None
    _table = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._tskv = TSKV()
        self._parser = self._parser_cls()
        self._buffer = deque()
        self._pusher = ClickhouseTSKVPusher(loger, self._table)
        self._last_processed_datetime = None

    def process_lines(self, lines):
        count = 0
        for line in lines:
            line = line.strip()
            done = self.process_line(line)
            if done:
                count += 1
        return count

    def last_processed_datetime(self):
        return self._last_processed_datetime

    def process_line(self, line):
        try:
            data = self._tskv.to_dict(line)
            return self.process(data)
        except ValueError:
            return False

    def process(self, data):
        record = self._parser.parse(data)
        if not record:
            return False
        if not record.valid:
            return False
        self._buffer.append(record)
        self._last_processed_datetime = record.timestamp
        return True

    def flush(self, force=False):
        loger.debug("Flushing %d records to database", len(self._buffer))
        self._pusher.push(self._buffer)


class LogProcessor(BaseProcessor):
    _parser_cls = LogParser
    _table = "feedback_events"  # rename table
