import logging
import json
from io import BytesIO
from collections import defaultdict
from django.conf import settings
from fan.db import clickhouse
from fan.models import StatsCampaign
from fan.utils.cached_getters import get_campaign
from fan.utils.clickhouse import connect_clickhouse
from ._utils import int_or_none, int_or_zero


class StatbagLoader:
    def __init__(self):
        self.clickhouse = connect_clickhouse()
        self._campaign_stats_changes = set()

    def update_stats(self, date):
        campaigns = self._load_updated_campaigns_by_date(date)
        logging.info("campaigns to process: %s (%d total)", json.dumps(campaigns), len(campaigns))
        stats = self._load_aggregated_stats_for_campaigns(campaigns)
        self._update_campaign_stats_unique(
            stats,
        )

    def _load_updated_campaigns_by_date(self, date):
        sql = """
            SELECT distinct campaign
            FROM {database}.feedback_events
            WHERE date='{date}'
        """
        result = self.clickhouse.query(
            sql.format(
                database=settings.CLICKHOUSE_DATABASE,
                date=date.strftime("%Y-%m-%d"),
            )
        )
        campaigns = []
        for line in BytesIO(result):
            (campaign,) = clickhouse.unquote_row(line)
            campaign = int_or_none(campaign)
            if campaign != None:
                campaigns.append(campaign)
        return campaigns

    def _load_aggregated_stats_for_campaigns(self, campaigns):
        sql = """
            SELECT
                event,
                campaign,
                count(distinct recipient)
            FROM
                {database}.feedback_events GLOBAL INNER JOIN campaigns USING campaign
            WHERE
                date >= now() - INTERVAL 14 DAY = 1
            GROUP BY event, campaign;
        """
        temp_struct = b"campaign UInt64"
        temp_data = BytesIO()
        for campaign in campaigns:
            temp_data.write(clickhouse.quote_row(campaign))
        temp_data.seek(0)

        query = clickhouse.Query(
            sql.format(database=settings.CLICKHOUSE_DATABASE),
            external_name="campaigns",
            external_file=temp_data,
            structure=temp_struct,
        )
        result = self.clickhouse.query(query)

        del temp_data

        stats = defaultdict(lambda: {k: 0 for k in ("pixel", "unsubscribe")})
        for line in BytesIO(result):
            event, campaign, count = clickhouse.unquote_row(line)
            event = event.decode()
            campaign = int_or_none(campaign)
            count = int_or_zero(count)
            stats[campaign][event] = count
        return stats

    def _update_campaign_stats_unique(self, stats):
        for campaign, stat in stats.items():
            if get_campaign(campaign, raise_if_needed=False):
                cur_stats, _ = StatsCampaign.objects.get_or_create(
                    campaign_id=campaign,
                )
                cur_stats.reads = stat.get("pixel", 0)
                cur_stats.unsubscribes = stat.get("unsubscribe", 0)
                cur_stats.save()
            else:
                logging.warning("no such campaign %d", campaign)
