import logging
from os.path import isfile

from django.conf import settings
from django.core.management.base import BaseCommand

from fan.utils.logging_ import duplicate_log_to_stream
from stats.logbrocker_workaround.clickhouse import LogProcessor
from stats.logbrocker_workaround.utils import get_log_lines_since


class Command(BaseCommand):
    help = "Upload TSKV logs in dev and test environments to clickhouse"

    def add_arguments(self, parser):
        parser.add_argument("--log-file", dest="log_file", default=settings.TYPEDLOG_FILENAME)
        parser.add_argument("--log-date", dest="log_date", default=None)

    def handle(self, *args, **options):
        with duplicate_log_to_stream():
            try:
                logging.info("loading log to CH")
                logfile = options["log_file"]
                date = options["log_date"]

                if not date:
                    date = self._try_restore_date_from_position_file(logfile)

                lines = get_log_lines_since(logfile, date)
                pusher = LogProcessor()
                count = pusher.process_lines(lines)
                pusher.flush()

                if pusher.last_processed_datetime():
                    self._write_position_to_file(logfile, pusher.last_processed_datetime())

                logging.info("processed %s lines from log", count)
            except Exception:
                logging.exception("failed to load log to ClickHouse")

    def _try_restore_date_from_position_file(self, logfile):
        str = self._read_str_from_file(self._position_file_name(logfile))
        return str if len(str) > 0 else None

    def _write_position_to_file(self, logfile, position_date_str):
        str = self._write_str_to_file(self._position_file_name(logfile), position_date_str)

    def _position_file_name(self, logfile):
        return logfile + ".position"

    def _write_str_to_file(self, filename, str):
        with open(filename, "w") as file:
            file.write(str)

    def _read_str_from_file(self, filename):
        if not isfile(filename):
            return ""
        with open(filename, "r") as file:
            return file.read().replace("\n", "").strip()
