import logging

from django.conf import settings
from fan.utils.logging_ import (
    LockedHandler,
    encode_complex_or_none,
    inject_timestamp_hack,
    DEFAULT_EXCLUDED_FIELDS,
    DEFAULT_ORDERED_FIRST,
    DEFAULT_RENAME_FIELDS,
    TSKVFormatter,
)
from logging.handlers import RotatingFileHandler

_typed_log_handler = None


def log_email_unsubscribe(
    email, campaign_id, letter_id, message_id, for_testing, error_code=None, error_text=None
):
    data = {
        "event": "unsubscribe",
        "email": email,
        "campaign": campaign_id,
        "letter": letter_id,
        "message": message_id,
        "test": for_testing,
        "error_code": error_code,
        "error_text": error_text,
    }
    _log(data)


def log_pixel(
    email,
    campaign_id,
    letter_id,
    message_id,
    for_testing,
    user_agent,
    user_ip,
    error_code=None,
    error_text=None,
):
    data = {
        "event": "pixel",
        "email": email,
        "campaign": campaign_id,
        "letter": letter_id,
        "message": message_id,
        "test": for_testing,
        "user_agent": user_agent,
        "user_ip": user_ip,
        "error_code": error_code,
        "error_text": error_text,
    }
    _log(data)


def set_typed_log_handler(handler):
    global _typed_log_handler
    _typed_log_handler = _create_typed_log_handler(handler)
    return _typed_log_handler


def _log(data):
    flat_data = encode_complex_or_none(data.copy())
    flat_data = inject_timestamp_hack(flat_data)
    handler = _get_typed_log_handler()
    handler.emit(logging.makeLogRecord(flat_data))


def _get_typed_log_handler():
    global _typed_log_handler
    if _typed_log_handler is None:
        _typed_log_handler = _create_typed_log_handler(_create_base_log_handler())
    return _typed_log_handler


def _create_base_log_handler():
    return RotatingFileHandler(filename=settings.TYPEDLOG_FILENAME)


def _create_typed_log_handler(handler):
    fmt = {
        "tskv_format": settings.TYPEDLOG_TSKV_FORMAT,
        "exclude_fields": DEFAULT_EXCLUDED_FIELDS
        | {"threadName", "levelname", "message", "caller_info"},
        "ordered_first": DEFAULT_ORDERED_FIRST,
        "rename_fields": DEFAULT_RENAME_FIELDS,
    }
    handler.setFormatter(TSKVFormatter(**fmt))
    return LockedHandler(handler)
