#!/usr/bin/env python3
from http import HTTPStatus
from http.server import SimpleHTTPRequestHandler
from json import dumps, load
import socketserver
import socket
import time
import datetime
import argparse
import pathlib


LISTEN_ADDR = "::"

RESPONSE_DELAY = 0.05
DEFAULT_ERROR_CODE = 404
DEFAULT_ERROR_MESSAGE = "not_found"


class Handler(SimpleHTTPRequestHandler):
    def __init__(self, *args, **kwargs):
        self.duration = RESPONSE_DELAY
        with open(URLS_RESPONSES_FILE_PATH, "r") as f:
            self.urls_responses = load(f)
        super().__init__(*args, **kwargs)

    def serve_json(self, data):
        self.resp = dumps(data)
        self.send_response(200)
        self.send_header("Content-Length", len(self.resp))
        self.send_header("Content-Type", "application/json")
        self.end_headers()
        self.wfile.write(self.resp.encode())

    def log_request(self, code):
        resp = getattr(self, "resp", "-")
        self.log_message(code, "response=%s", resp)

    def log_error(self, format, code, message):
        self.log_message(code, "message=%s", message)

    def log_message(self, code, format, *args):
        if isinstance(code, HTTPStatus):
            code = code.value
        print(
            "addr=%s\ttimestamp=%s\tmethod=%s\trequest=%s\tduration=%.3f\tstatus_code=%s\t%s"
            % (
                self.address_string(),
                self.log_date_time_string(),
                self.command,
                self.path,
                self.duration,
                str(code),
                format % args,
            )
        )

    def log_date_time_string(self):
        d = datetime.datetime.now()
        return d.strftime("%Y-%m-%d %H:%M:%S.%f")

    def do_GET(self):
        time.sleep(RESPONSE_DELAY)
        response = self.urls_responses.get(self.path)
        if response:
            self.serve_json(response)
        else:
            self.send_error(code=DEFAULT_ERROR_CODE, message=DEFAULT_ERROR_MESSAGE)


class MyTCPServer(socketserver.ThreadingTCPServer):
    address_family = socket.AF_INET6

    def server_bind(self):
        self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.socket.bind(self.server_address)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--datapath", type=pathlib.Path, required=True, help="A path to json file with responses"
    )
    parser.add_argument("--port", type=int, default=8000)
    return parser.parse_args()


def main():
    args = parse_args()
    if args.datapath:
        global URLS_RESPONSES_FILE_PATH
        URLS_RESPONSES_FILE_PATH = args.datapath
    port = args.port
    httpd = MyTCPServer((LISTEN_ADDR, port), Handler)
    print("starting server on port={}...".format(port))
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        pass
    httpd.server_close()


if __name__ == "__main__":
    main()
