#pragma once

#include "sender.h"
#include <typed_log/typed_log.h>
#include <common/mail_message.h>
#include <ymod_smtpclient/call.h>
#include <yplatform/module.h>

namespace fan::delivery {

struct module
    : sender
    , yplatform::module
{

    using smtp_client = ymod_smtpclient::Call;
    using smtp_client_ptr = shared_ptr<ymod_smtpclient::Call>;
    using smtp_request = ymod_smtpclient::Request;
    using smtp_response = ymod_smtpclient::Response;

    string smtp_host;
    unsigned short smtp_port;
    smtp_client_ptr smtp;
    bool dry_run = false;
    string reverse_path;

    module(const yplatform::ptree& conf)
    {
        smtp_host = conf.get<string>("smtp.host");
        smtp_port = conf.get<unsigned short>("smtp.port");
        dry_run = conf.get<bool>("dry_run");
        reverse_path = conf.get<string>("reverse_path");
        smtp = find_module<smtp_client>("smtp_client");
    }

    void send(task_context_ptr ctx, const mail_message& message, const without_data_cb& cb) override
    {
        smtp_request req;
        req.address.host = smtp_host;
        req.address.port = smtp_port;
        req.mailfrom.email = reverse_path.size() ? reverse_path : message.from_email;
        req.rcpts.emplace_back(message.recipient);
        req.message = message.eml;
        if (dry_run)
        {
            typed::log_mail_message_sent(
                ctx, "skip", "sent to /dev/null", "", message, reverse_path);
            return cb({});
        }
        smtp->asyncRun(
            ctx,
            req,
            [this, capture_self, ctx, cb, message](error_code err, smtp_response response) {
                string status = err ? "error" : "success";
                string reason = err ? err.message() : "";
                string smtp_response = response.session ? response.session->data : "";
                typed::log_mail_message_sent(
                    ctx, status, reason, smtp_response, message, reverse_path);
                cb(err);
            });
    }
};

}
