#pragma once

#include <common/types.h>
#include <yplatform/encoding/base64.h>
#include <mimeparser/part.h>
#include <boost/algorithm/string/replace.hpp>

namespace fan::rendering {

namespace detail {

inline void replace_params(string& str, const map<string, string>& params)
{
    for (auto&& [key, value] : params)
    {
        boost::algorithm::replace_all(str, "%" + key + "%", value);
    }
}

struct part_meta
{
    size_t body_offset = 0;
    size_t body_len = 0;
};

inline bool operator<(const part_meta& lhs, const part_meta& rhs)
{
    return lhs.body_offset < rhs.body_offset;
}

inline void trim_part(const string& eml, part_meta& part)
{
    while (part.body_len && (eml[part.body_offset] == '\r' || eml[part.body_offset] == '\n'))
    {
        ++part.body_offset;
        --part.body_len;
    }
    while (part.body_len &&
           (eml[part.body_offset + part.body_len - 1] == '\r' ||
            eml[part.body_offset + part.body_len - 1] == '\n'))
    {
        --part.body_len;
    }
}

inline void collect_base64_text_parts_impl(
    const string& eml,
    const mulca_mime::part& mime,
    vector<part_meta>& parts)
{
    for (auto&& subpart : mime.parts)
    {
        collect_base64_text_parts_impl(eml, subpart, parts);
    }
    if (mime.header.content_type.type == "text" &&
        mime.header.content_transfer_encoding == "base64")
    {
        part_meta part{ .body_offset = mime.body.offset(), .body_len = mime.body.length() };
        // Remove trimming line breaks from meta to preserve them on part modifications
        trim_part(eml, part);
        parts.push_back(part);
    }
}

inline vector<part_meta> collect_base64_text_parts(const string& eml, const mulca_mime::part& mime)
{
    vector<part_meta> parts;
    collect_base64_text_parts_impl(eml, mime, parts);
    return parts;
}

using transformer_type = function<string(string part_body)>;

inline void transform_parts(
    string& eml,
    vector<part_meta> parts,
    const transformer_type& transformer)
{
    if (parts.empty()) return;
    string eml_copy;
    eml_copy.reserve(eml.length());
    size_t last_processed = 0;
    std::sort(parts.begin(), parts.end());
    for (auto&& part : parts)
    {
        eml_copy += eml.substr(last_processed, part.body_offset - last_processed);
        eml_copy += transformer(eml.substr(part.body_offset, part.body_len));
        last_processed = part.body_offset + part.body_len;
    }
    eml_copy += eml.substr(last_processed);
    eml = eml_copy;
}

inline mulca_mime::part parse_mime(const string& eml)
{
    mulca_mime::part mime(eml, nullptr);
    mime.parse(0, eml.size());
    return mime;
}

inline void remove_line_breaks(string& str)
{
    boost::algorithm::replace_all(str, "\r\n", "");
    boost::algorithm::replace_all(str, "\n", "");
}

inline void add_line_breaks(string& str)
{
    static const size_t LINE_LENGTH = 76;
    string str_copy;
    str_copy.reserve(str.length());
    for (size_t pos = 0; pos < str.length(); pos += LINE_LENGTH)
    {
        if (str_copy.length()) str_copy += "\r\n";
        str_copy += str.substr(pos, LINE_LENGTH);
    }
    str = str_copy;
}

inline void replace_params_in_base64_text_parts(string& eml, const map<string, string>& params)
{
    auto mime = parse_mime(eml);
    auto parts = collect_base64_text_parts(eml, mime);
    transform_parts(eml, parts, [&params](string body) {
        remove_line_breaks(body);
        body = yplatform::base64_decode_str(body);
        replace_params(body, params);
        body = yplatform::base64_encode_str(body);
        add_line_breaks(body);
        return body;
    });
}

}

inline string_ptr replace_params(const string& eml_template, const map<string, string>& params)
{
    auto eml = make_shared<string>(eml_template);
    detail::replace_params(*eml, params);
    detail::replace_params_in_base64_text_parts(*eml, params);
    return eml;
}

}
