#pragma once

#include "polling_loop.h"
#include <tasks/send_tasks.h>
#include <delivery/sender.h>
#include <common/types.h>
#include <ymod_lease/lock_manager.h>
#include <ymod_lease/ymod_lease.h>
#include <yplatform/module.h>

namespace fan::send {

struct module : yplatform::module
{
    using lock_manager_type = ylease::lock_manager<ylease::node>;

    io_service& io;
    settings settings;
    shared_ptr<tasks::send_tasks> tasks;
    shared_ptr<delivery::sender> sender;
    size_t groups_count;

    shared_ptr<lock_manager_type> lock_manager;
    vector<polling_loop_ptr> loops;

    module(io_service& io) : io(io)
    {
    }

    void init(const yplatform::ptree& conf)
    {
        settings = settings::from_ptree(conf);
        tasks = find_module<tasks::send_tasks>("tasks");
        sender = find_module<delivery::sender>("delivery");
        if (settings.lease)
        {
            auto lease_node = find_module<ymod_lease::node>("lease_node");
            lock_manager = make_shared<lock_manager_type>(
                &io,
                lease_node,
                settings.lease->max_owned_groups,
                settings.lease->extra_acquire_count);
            lock_manager->init(
                io.wrap(weak_bind(&module::on_acquire_group, shared_from(this), ph::_1)),
                io.wrap(weak_bind(&module::on_release_group, shared_from(this), ph::_1)));
            groups_count = settings.lease->groups_count;
        }
        else
        {
            groups_count = 1;
        }
        loops.resize(groups_count, nullptr);
    }

    void start()
    {
        if (settings.lease)
        {
            add_groups_to_lock_manager();
        }
        else
        {
            start_loop(0);
        }
    }

    void add_groups_to_lock_manager()
    {
        vector<string> resources;
        for (size_t group_num = 0; group_num < settings.lease->groups_count; ++group_num)
        {
            resources.emplace_back(get_resource(group_num));
        }
        lock_manager->on_add_resources(resources);
    }

    string get_resource(size_t group_num)
    {
        return settings.lease->send_resource_prefix + std::to_string(group_num);
    }

    size_t parse_group_num(const string& resource)
    {
        size_t prefix_len = settings.lease->send_resource_prefix.length();
        size_t group_num = std::stoull(resource.substr(prefix_len));
        return group_num;
    }

    void on_acquire_group(const string& resource)
    {
        YLOG_L(info) << "group \"" << resource << "\" acquired";
        size_t group_num = parse_group_num(resource);
        start_loop(group_num);
    }

    void on_release_group(const string& resource)
    {
        YLOG_L(info) << "group \"" << resource << "\" released";
        size_t group_num = parse_group_num(resource);
        stop_loop(group_num);
    }

    void start_loop(size_t group_num)
    {
        if (loops[group_num])
        {
            YLOG_L(warning) << "start loop for group " << group_num << ", but it already started";
            return;
        }
        auto ctx = boost::make_shared<task_context>();
        LINFO_(ctx) << "start polling loop for send; group_num=" << group_num;
        auto loop = std::make_shared<polling_loop>(
            io, ctx, group_num, groups_count, tasks, sender, settings.polling);
        loops[group_num] = loop;
        yplatform::spawn(io, loop);
    }

    void stop_loop(size_t group_num)
    {
        if (!loops[group_num]) return;
        LINFO_(loops[group_num]->task_ctx) << "stop polling loop";
        loops[group_num]->stop();
        loops[group_num].reset();
    }
};
}
