#pragma once

#include "send_campaign_op.h"
#include "settings.h"
#include <tasks/send_tasks.h>
#include <delivery/module.h>
#include <typed_log/typed_log.h>
#include <common/campaign.h>
#include <common/types.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace fan::send {

template <typename SendOp, typename Tasks, typename Delivery, typename Timer>
struct polling_loop_impl
    : std::enable_shared_from_this<polling_loop_impl<SendOp, Tasks, Delivery, Timer>>
{
    using yield_context = yplatform::yield_context<polling_loop_impl>;

    io_service& io;
    Timer timer;
    task_context_ptr task_ctx;
    size_t group_num;
    size_t groups_count;
    polling_settings settings;
    shared_ptr<Tasks> tasks;
    shared_ptr<Delivery> delivery;

    error_code err;
    vector<campaign> campaigns;
    size_t campaign_num = 0;
    size_t attempt_num = 0;
    duration next_retry_interval;
    bool stopped = false;

    polling_loop_impl(
        io_service& io,
        task_context_ptr task_ctx,
        size_t group_num,
        size_t groups_count,
        shared_ptr<Tasks> tasks,
        shared_ptr<Delivery> delivery,
        const polling_settings& settings)
        : io(io)
        , timer(io)
        , task_ctx(task_ctx)
        , group_num(group_num)
        , groups_count(groups_count)
        , settings(settings)
        , tasks(tasks)
        , delivery(delivery)
    {
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            while (!stopped)
            {
                yield wait(settings.interval, yield_ctx.capture(err));
                if (err) break;
                yield tasks->get_pending_campaigns(
                    task_ctx, fetch_count(), yield_ctx.capture(err, campaigns));
                if (err)
                {
                    LERR_(task_ctx) << "get pending task error: " << err.message();
                    continue;
                }
                remove_not_owned_groups(campaigns);
                for (campaign_num = 0; campaign_num < campaigns.size(); ++campaign_num)
                {
                    attempt_num = 0;
                    next_retry_interval = settings.interval;
                    do
                    {
                        if (attempt_num)
                        {
                            next_retry_interval *= settings.backoff_multiplier;
                            yield wait(next_retry_interval, yield_ctx.capture(err));
                        }
                        if (stopped) yield break;
                        yield send_campaign(campaigns[campaign_num], yield_ctx.capture(err));
                        ++attempt_num;
                    } while (attempt_num <= settings.max_retries && err);
                    if (err)
                    {
                        yield tasks->mark_campaign_failed(
                            task_ctx,
                            campaigns[campaign_num],
                            err.message(),
                            yield_ctx.capture(err));
                        // Ignore error
                    }
                }
            }
        }
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& err)
        {
            LERR_(task_ctx) << "polling_loop exception: " << err.what();
            yplatform::spawn(io, shared_from(this));
        }
    }

    void stop()
    {
        stopped = true;
    }

    template <typename Handler>
    void wait(const duration& interval, const Handler& handler)
    {
        timer.expires_from_now(interval);
        timer.async_wait(handler);
    }

    template <typename Handler>
    void send_campaign(const campaign& campaign, const Handler& handler)
    {
        auto op = std::make_shared<SendOp>(task_ctx, campaign, tasks, delivery, handler);
        yplatform::spawn(op);
    }

    size_t fetch_count()
    {
        return 2 * groups_count;
    }

    void remove_not_owned_groups(vector<campaign>& campaigns)
    {
        campaigns.erase(
            std::remove_if(
                campaigns.begin(),
                campaigns.end(),
                [this](auto&& campaign) { return !belongs_to_group(campaign); }),
            campaigns.end());
    }

    bool belongs_to_group(const campaign& campaign)
    {
        size_t campaign_group = campaign.id % groups_count;
        return campaign_group == group_num;
    }
};

using polling_loop = polling_loop_impl<
    send_campaign_op<tasks::send_tasks, delivery::sender>,
    tasks::send_tasks,
    delivery::sender,
    timer>;
using polling_loop_ptr = shared_ptr<polling_loop>;
}

#include <yplatform/unyield.h>
