#pragma once

#include "send_segment_op.h"
#include "split_into_segments.h"
#include <rendering/replace_params.h>
#include <tasks/test_tasks.h>
#include <delivery/module.h>
#include <typed_log/typed_log.h>
#include <common/campaign.h>
#include <common/recipient_data.h>
#include <common/send_stats.h>
#include <common/validation.h>
#include <common/errors.h>
#include <common/types.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace fan::send {

static const size_t SEGMENTS_MAX_COUNT = 8;

template <typename Tasks, typename Delivery>
struct send_campaign_op : std::enable_shared_from_this<send_campaign_op<Tasks, Delivery>>
{
    using yield_context = yplatform::yield_context<send_campaign_op>;

    task_context_ptr task_ctx;
    campaign campaign;
    shared_ptr<Tasks> tasks;
    shared_ptr<Delivery> delivery;
    without_data_cb cb;

    error_code err;
    string_ptr eml_template;
    string_ptr eml;
    vector<recipient_data> recipients;
    vector<string> unsubscribe_list;
    vector<map<string, string>> template_params;

    send_stats stats;
    vector<vector<recipient_data>> recipient_segments;
    atomic<size_t> smtp_sent_count = 0;

    send_campaign_op(
        task_context_ptr task_ctx,
        const struct campaign& campaign,
        shared_ptr<Tasks> tasks,
        shared_ptr<Delivery> delivery,
        const without_data_cb& cb)
        : task_ctx(task_ctx), campaign(campaign), tasks(tasks), delivery(delivery), cb(cb)
    {
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            yield tasks->get_campaign_recipients(
                task_ctx, campaign, yield_ctx.capture(err, recipients));
            if (err)
            {
                log_error("get recipients error: " + err.message());
                yield break;
            }
            if (recipients.empty())
            {
                err = error::no_recipients;
                log_error("no recipients");
                yield break;
            }
            yield tasks->get_campaign_unsubscribe_list(
                task_ctx, campaign, yield_ctx.capture(err, unsubscribe_list));
            if (err)
            {
                log_error("get unsubscribe_list error: " + err.message());
                yield break;
            }

            stats.uploaded_count = recipients.size();
            stats.invalid_count = remove_invalid_emails(recipients);
            stats.duplicated_count = remove_duplicated(recipients);
            stats.unsubscribed_count = remove_unsubscribed(recipients, unsubscribe_list);

            yield tasks->get_campaign_eml_template(
                task_ctx, campaign, yield_ctx.capture(err, eml_template));
            if (err)
            {
                log_error("get eml template error: " + err.message());
                yield break;
            }

            // TODO Try to replace params in eml template for first recipient before mark sent
            yield tasks->mark_campaign_sent(task_ctx, campaign, stats, yield_ctx.capture(err));
            if (err)
            {
                log_error("mark campaign sent error: " + err.message());
                yield break;
            }

            recipient_segments = split_into_segments(recipients, SEGMENTS_MAX_COUNT);
            yield send_campaign().then(yield_ctx);

            err = error_code();
            log_success();
        }
        if (yield_ctx.is_complete())
        {
            cb(err);
        }
    }

    auto send_campaign()
    {
        vector<future<void>> futures;
        for (size_t segment_idx = 0; segment_idx < recipient_segments.size(); ++segment_idx)
        {
            auto future = run_send_segment(segment_idx);
            futures.push_back(future);
        }
        return yplatform::future::future_multi_and(futures);
    }

    future<void> run_send_segment(size_t segment_idx)
    {
        promise<void> send_segment_promise;

        auto op = std::make_shared<send_segment_op<Tasks, Delivery>>(
            task_ctx,
            campaign,
            tasks,
            eml_template,
            recipient_segments[segment_idx],
            delivery,
            [this, capture_self, send_segment_promise](error_code, size_t sent_count) mutable {
                // Ignore error
                smtp_sent_count += sent_count;
                send_segment_promise.set();
            });
        yplatform::spawn(op);

        return send_segment_promise;
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& err)
        {
            log_error(err.what());
        }
        cb(error::send_error);
    }

    size_t remove_invalid_emails(vector<recipient_data>& recipients)
    {
        size_t size_before_remove = recipients.size();
        recipients.erase(
            std::remove_if(
                recipients.begin(),
                recipients.end(),
                [](auto&& recipient) { return !is_valid_email(recipient.email); }),
            recipients.end());
        return size_before_remove - recipients.size();
    }

    size_t remove_duplicated(vector<recipient_data>& recipients)
    {
        size_t size_before_remove = recipients.size();
        std::sort(recipients.begin(), recipients.end(), [](const auto& lhs, const auto& rhs) {
            return lhs.email < rhs.email;
        });
        recipients.erase(
            std::unique(
                recipients.begin(),
                recipients.end(),
                [](const auto& lhs, const auto& rhs) { return lhs.email == rhs.email; }),
            recipients.end());
        return size_before_remove - recipients.size();
    }

    size_t remove_unsubscribed(
        vector<recipient_data>& recipients,
        const vector<string>& unsubscribed)
    {
        size_t size_before_remove = recipients.size();
        std::unordered_set<string> unsubscribed_set(unsubscribed.begin(), unsubscribed.end());
        recipients.erase(
            std::remove_if(
                recipients.begin(),
                recipients.end(),
                [&](auto&& recipient) { return unsubscribed_set.count(recipient.email); }),
            recipients.end());
        return size_before_remove - recipients.size();
    }

    void log_error(const string& reason)
    {
        typed::log_campaign_processed(task_ctx, campaign, "error", reason);
    }

    void log_success()
    {
        typed::log_campaign_processed(
            task_ctx,
            campaign,
            "success",
            "",
            { { "uploaded_count", std::to_string(stats.uploaded_count) },
              { "invalid", std::to_string(stats.invalid_count) },
              { "duplicated", std::to_string(stats.duplicated_count) },
              { "unsubscribed", std::to_string(stats.unsubscribed_count) },
              { "smtp_sent", std::to_string(smtp_sent_count) } });
    }
};

}

#include <yplatform/unyield.h>
