#pragma once

#include <rendering/replace_params.h>
#include <delivery/module.h>
#include <typed_log/typed_log.h>
#include <common/campaign.h>
#include <common/recipient_data.h>
#include <common/errors.h>
#include <common/types.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace fan::send {

template <typename Tasks, typename Delivery>
struct send_segment_op : std::enable_shared_from_this<send_segment_op<Tasks, Delivery>>
{
    using yield_context = yplatform::yield_context<send_segment_op>;

    task_context_ptr task_ctx;
    campaign campaign;
    shared_ptr<Tasks> tasks;
    string_ptr eml_template;
    vector<recipient_data> recipients;
    shared_ptr<Delivery> delivery;
    size_t_cb cb;

    size_t recipient_num = 0;
    size_t smtp_sent_count = 0;
    error_code err;

    send_segment_op(
        task_context_ptr task_ctx,
        const struct campaign& campaign,
        shared_ptr<Tasks> tasks,
        string_ptr eml_template,
        const vector<recipient_data>& recipients,
        shared_ptr<Delivery> delivery,
        const size_t_cb& cb)
        : task_ctx(task_ctx)
        , campaign(campaign)
        , tasks(tasks)
        , eml_template(eml_template)
        , recipients(recipients)
        , delivery(delivery)
        , cb(cb)
    {
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            yield tasks->get_campaign_template_params(
                task_ctx, campaign, recipients, yield_ctx.capture(err, recipients));
            if (err)
            {
                log_error("get template params error: " + err.message());
                yield break;
            }

            for (recipient_num = 0; recipient_num < recipients.size(); ++recipient_num)
            {
                yield send(
                    campaign, eml_template, recipients[recipient_num], yield_ctx.capture(err));
                // Ignore error
                if (!err) ++smtp_sent_count;
            }
            log_success();
        }
        if (yield_ctx.is_complete())
        {
            cb(error_code(), smtp_sent_count);
        }
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& err)
        {
            log_error(err.what());
        }
        cb(error::send_error, smtp_sent_count);
    }

    template <typename Handler>
    void send(
        const struct campaign& campaign,
        string_ptr eml_template,
        const recipient_data& recipient,
        const Handler& handler)
    {
        mail_message message = { .from_email = campaign.from_email,
                                 .recipient = recipient.email,
                                 .eml = rendering::replace_params(
                                     *eml_template, recipient.template_params),
                                 .account_slug = campaign.account_slug,
                                 .campaign_slug = campaign.campaign_slug };
        delivery->send(task_ctx, message, handler);
    }

    void log_error(const string& reason)
    {
        typed::log_segment_processed(
            task_ctx,
            campaign,
            "error",
            reason,
            { { "segment_size", std::to_string(recipients.size()) },
              { "smtp_sent", std::to_string(smtp_sent_count) } });
    }

    void log_success()
    {
        typed::log_segment_processed(
            task_ctx,
            campaign,
            "success",
            "",
            { { "segment_size", std::to_string(recipients.size()) },
              { "smtp_sent", std::to_string(smtp_sent_count) } });
    }
};

}

#include <yplatform/unyield.h>
