#pragma once

#include "send_tasks.h"
#include "test_tasks.h"
#include <common/errors.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>

namespace fan::tasks {

template <typename HttpClient>
struct module_impl
    : test_tasks
    , send_tasks
    , yplatform::module
{
    shared_ptr<HttpClient> http_client;

    module_impl() : module_impl(find_module<HttpClient>("send_api_client"))
    {
    }

    module_impl(shared_ptr<HttpClient> http_client) : http_client(http_client)
    {
    }

    void get_pending_task(task_context_ptr ctx, const pending_task_cb& cb) override
    {
        auto req = yhttp::request::GET("/api/send/test-task-list?count=1");
        http_client->async_run(ctx, req, [this, capture_self, ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "pending task error: bad HTTP status " << response.status << " "
                           << response.body;
                return cb(error::tasks_api_error, {});
            }
            optional<test_send_task> ret;
            if (auto error = parse_test_task_response(response, ret))
            {
                LERR_(ctx) << "parse pending task response error: " << *error;
                return cb(error::tasks_api_error, {});
            }
            cb({}, ret);
        });
    }

    void get_task_eml(
        task_context_ptr ctx,
        const string& task_id,
        const string& recipient,
        const string_ptr_cb& cb) override
    {
        auto req = yhttp::request::GET(
            "/api/send/test-task-eml" +
            yhttp::url_encode({ { "task_id", task_id }, { "recipient", recipient } }));
        http_client->async_run(ctx, req, [ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "task eml error: bad HTTP status " << response.status << " "
                           << response.body;
                return cb(error::tasks_api_error, {});
            }
            cb({}, make_shared<string>(std::move(response.body)));
        });
    }

    void complete_task(task_context_ptr ctx, const string& task_id, const without_data_cb& cb)
        override
    {
        auto req = yhttp::request::DELETE("/api/send/test-task?task_id=" + task_id);
        http_client->async_run(ctx, req, [ctx, cb](auto err, auto response) {
            if (err) return cb(err);
            if (response.status != 200)
            {
                LERR_(ctx) << "complete task error: bad HTTP status " << response.status << " "
                           << response.body;
                return cb(error::tasks_api_error);
            }
            cb({});
        });
    }

    void get_pending_campaigns(task_context_ptr ctx, size_t count, const campaigns_cb& cb) override
    {
        auto req = yhttp::request::GET(
            "/api/send/pending-campaign-list" +
            yhttp::url_encode({ { "count", std::to_string(count) } }));
        http_client->async_run(ctx, req, [this, capture_self, ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "pending campaign list error: bad HTTP status " << response.status
                           << " " << response.body;
                return cb(error::tasks_api_error, {});
            }
            vector<campaign> campaigns;
            if (auto error = parse_campaigns_response(response, campaigns))
            {
                LERR_(ctx) << "parse pending campaign list response error: " << *error;
                return cb(error::tasks_api_error, {});
            }
            cb({}, campaigns);
        });
    }

    void get_campaign_recipients(
        task_context_ptr ctx,
        const campaign& campaign,
        const recipients_cb& cb) override
    {
        auto req = yhttp::request::GET(
            "/api/send/campaign-recipient-list" +
            yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                { "campaign_slug", campaign.campaign_slug } }));
        http_client->async_run(ctx, req, [this, capture_self, ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "campaign recipient list error: bad HTTP status " << response.status
                           << " " << response.body;
                return cb(error::tasks_api_error, {});
            }
            vector<recipient_data> recipients;
            if (auto error = parse_recipient_data_array_response(response, recipients))
            {
                LERR_(ctx) << "parse recipient list error: " << *error;
                return cb(error::tasks_api_error, {});
            }
            cb({}, recipients);
        });
    }

    void get_campaign_unsubscribe_list(
        task_context_ptr ctx,
        const campaign& campaign,
        const strings_cb& cb) override
    {
        auto req = yhttp::request::GET(
            "/api/send/campaign-unsubscribe-list" +
            yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                { "campaign_slug", campaign.campaign_slug } }));
        http_client->async_run(ctx, req, [this, capture_self, ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "campaign unsubscribe list error: bad HTTP status " << response.status
                           << " " << response.body;
                return cb(error::tasks_api_error, {});
            }
            vector<string> unsubscribed;
            if (auto error = parse_string_array_response(response, unsubscribed))
            {
                LERR_(ctx) << "parse unsubscribe list error: " << *error;
                return cb(error::tasks_api_error, {});
            }
            cb({}, unsubscribed);
        });
    }

    void get_campaign_eml_template(
        task_context_ptr ctx,
        const campaign& campaign,
        const string_ptr_cb& cb) override
    {
        auto req = yhttp::request::GET(
            "/api/send/campaign-eml-template" +
            yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                { "campaign_slug", campaign.campaign_slug } }));
        http_client->async_run(ctx, req, [ctx, cb](auto err, auto response) {
            if (err) return cb(err, {});
            if (response.status != 200)
            {
                LERR_(ctx) << "campaign eml error: bad HTTP status " << response.status << " "
                           << response.body;
                return cb(error::tasks_api_error, {});
            }
            cb({}, make_shared<string>(std::move(response.body)));
        });
    }

    void get_campaign_template_params(
        task_context_ptr ctx,
        const campaign& campaign,
        const vector<recipient_data>& recipients,
        const recipients_cb& cb) override
    {
        auto req = make_get_request(
            "/api/send/campaign-template-params" +
                yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                    { "campaign_slug", campaign.campaign_slug } }),
            make_recipients_json(recipients));
        http_client->async_run(
            ctx,
            req,
            [this, capture_self, ret = recipients, ctx, cb](auto err, auto response) mutable {
                if (err) return cb(err, ret);
                if (response.status != 200)
                {
                    LERR_(ctx) << "get campaign template params error: bad HTTP status "
                               << response.status << " " << response.body;
                    return cb(error::tasks_api_error, ret);
                }
                if (auto error = parse_template_params_response(response, ret))
                {
                    LERR_(ctx) << "parse campaign template params response error: " << *error;
                    return cb(error::tasks_api_error, ret);
                }
                cb({}, ret);
            });
    }

    void mark_campaign_sent(
        task_context_ptr ctx,
        const campaign& campaign,
        const send_stats& stats,
        const without_data_cb& cb) override
    {
        auto req = make_post_request(
            "/api/send/campaign-state" +
                yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                    { "campaign_slug", campaign.campaign_slug },
                                    { "state", "sent" } }),
            make_stats_json(stats));
        http_client->async_run(ctx, req, [ctx, cb](auto err, auto response) {
            if (err) return cb(err);
            if (response.status != 200)
            {
                LERR_(ctx) << "campaign mark as sent error: bad HTTP status " << response.status
                           << " " << response.body;
                return cb(error::tasks_api_error);
            }
            cb({});
        });
    }

    void mark_campaign_failed(
        task_context_ptr ctx,
        const campaign& campaign,
        const string& reason,
        const without_data_cb& cb) override
    {
        auto req = make_post_request(
            "/api/send/campaign-state" +
                yhttp::url_encode({ { "account_slug", campaign.account_slug },
                                    { "campaign_slug", campaign.campaign_slug },
                                    { "state", "failed" } }),
            make_error_json(reason));
        http_client->async_run(ctx, req, [ctx, cb](auto err, auto response) {
            if (err) return cb(err);
            if (response.status != 200)
            {
                LERR_(ctx) << "campaign mark as sent error: bad HTTP status " << response.status
                           << " " << response.body;
                return cb(error::tasks_api_error);
            }
            cb({});
        });
    }

    optional<string> parse_json_array_response(
        const yhttp::response& response,
        yplatform::json_value& res)
    {
        if (auto error = res.parse(response.body)) return error;
        if (!res.is_array()) return "response is not array";
        return {};
    }

    optional<string> parse_string_array_response(
        const yhttp::response& response,
        vector<string>& res)
    {
        yplatform::json_value json;
        if (auto error = json.parse(response.body)) return error;
        if (!json.is_array()) return "response is not array";
        for (auto it = json.array_begin(); it != json.array_end(); ++it)
        {
            res.emplace_back((*it).to_string());
        }
        return {};
    }

    optional<string> parse_recipient_data_array_response(
        const yhttp::response& response,
        vector<recipient_data>& res)
    {
        yplatform::json_value json;
        if (auto error = json.parse(response.body)) return error;
        if (!json.is_array()) return "response is not array";
        for (auto it = json.array_begin(); it != json.array_end(); ++it)
        {
            res.emplace_back(recipient_data::from_json(*it));
        }
        return {};
    }

    optional<string> parse_test_task_response(
        const yhttp::response& response,
        optional<test_send_task>& res)
    {

        yplatform::json_value json;
        if (auto error = parse_json_array_response(response, json)) return error;
        try
        {
            if (json.size()) res = test_send_task::from_json(*json.array_begin());
        }
        catch (const exception& e)
        {
            return e.what();
        }
        return {};
    }

    optional<string> parse_campaigns_response(
        const yhttp::response& response,
        vector<campaign>& res)
    {
        yplatform::json_value json;
        if (auto error = parse_json_array_response(response, json)) return error;
        try
        {
            for (auto it = json.array_begin(); it != json.array_end(); ++it)
            {
                res.emplace_back(campaign::from_json(*it));
            }
        }
        catch (const exception& e)
        {
            return e.what();
        }
        return {};
    }

    optional<string> parse_template_params_response(
        const yhttp::response& response,
        vector<recipient_data>& res)
    {
        yplatform::json_value json;
        if (auto error = json.parse(response.body)) return error;
        if (!json.is_array()) return "response is not array";
        if (json.size() != res.size())
        {
            std::stringstream error;
            error << "returned params count=" << json.size()
                  << " not correspond with requested recipients count=" << res.size();
            return error.str();
        }
        auto res_it = res.begin();
        for (auto json_it = json.array_begin(); json_it != json.array_end(); ++json_it, ++res_it)
        {
            auto&& recipient = *json_it;
            for (auto param_it = recipient.members_begin(); param_it != recipient.members_end();
                 ++param_it)
            {
                string key{ param_it.key() };
                string value = (*param_it).to_string();
                res_it->template_params[key] = value;
            }
        }
        return {};
    }

    yplatform::json_value make_recipients_json(const vector<recipient_data>& recipients)
    {
        yplatform::json_value recipients_json;
        recipients_json.set_array();
        for (auto&& recipient : recipients)
        {
            recipients_json.push_back(recipient.email);
        }
        yplatform::json_value ret;
        ret["recipients"] = recipients_json;
        return ret;
    }

    yplatform::json_value make_stats_json(const send_stats& stats)
    {
        yplatform::json_value ret;
        ret["email_uploaded"] = stats.uploaded_count;
        ret["email_unsubscribed"] = stats.unsubscribed_count;
        ret["email_duplicated"] = stats.duplicated_count;
        ret["email_invalid"] = stats.invalid_count;
        return ret;
    }

    yplatform::json_value make_error_json(const string& reason)
    {
        yplatform::json_value ret;
        ret["error"] = reason;
        return ret;
    }

    yhttp::request make_post_request(const string& url, const yplatform::json_value& body)
    {
        string headers = "Content-Type: application/json\r\n";
        auto req = yhttp::request::POST(url, headers, body.stringify());
        return req;
    }

    yhttp::request make_get_request(const string& url, const yplatform::json_value& body_json)
    {
        string body = body_json.stringify();
        std::stringstream headers;
        // XXX HTTP client does not add content-length header for get request
        headers << "Content-Type: application/json\r\n"
                << "Content-Length: " << body.length() << "\r\n";
        auto req = yhttp::request::GET(url, headers.str());
        req.body = boost::make_shared<string>(std::move(body));
        return req;
    }
};

using module = module_impl<yhttp::cluster_client>;

}
