from django.views.decorators.csrf import csrf_exempt
from rest_framework.authentication import SessionAuthentication
from rest_framework.parsers import JSONParser
from rest_framework.renderers import BrowsableAPIRenderer, JSONRenderer
from rest_framework.response import Response
from rest_framework.views import APIView

from fan.app import raven

from fan_ui.api.authentication import TVMAuthentication
from fan_ui.typed_log import log_api_response

# from .paginator import Paginator
from .permissions import NoPermission


DEFAULT_AUTHENTICATION = (
    TVMAuthentication,
    SessionAuthentication,
)


class Endpoint(APIView):
    authentication_classes = DEFAULT_AUTHENTICATION
    renderer_classes = (JSONRenderer, BrowsableAPIRenderer)
    parser_classes = (JSONParser,)
    permission_classes = (NoPermission,)
    pagination_class = None
    page_size = None

    def convert_args(self, request, *args, **kwargs):
        return args, kwargs

    def handle_exception(self, request, exc):
        try:
            return super().handle_exception(exc)
        except Exception:
            import sys
            import traceback

            sys.stderr.write(traceback.format_exc())
            event = raven.captureException(request=request)
            if event:
                event_id = raven.get_ident(event)
            else:
                event_id = None
            context = {
                "detail": "Internal Error",
                "errorId": event_id,
            }
            return Response(context, status=500)

    def create_audit_entry(self, request, **kwargs):
        pass

    @csrf_exempt
    def dispatch(self, request, *args, **kwargs):
        """
        Identical to rest framework's dispatch except we add the ability
        to convert arguments (for common URL params).
        """
        self.args = args
        self.kwargs = kwargs
        request = self.initialize_request(request, *args, **kwargs)
        self.request = request
        self.headers = self.default_response_headers  # deprecate?

        try:
            self.initial(request, *args, **kwargs)

            # Get the appropriate handler method
            if request.method.lower() in self.http_method_names:
                handler = getattr(self, request.method.lower(), self.http_method_not_allowed)

                (args, kwargs) = self.convert_args(request, *args, **kwargs)
                self.args = args
                self.kwargs = kwargs
            else:
                handler = self.http_method_not_allowed

            response = handler(request, *args, **kwargs)

        except Exception as exc:
            response = self.handle_exception(request, exc)

        self.response = self.finalize_response(request, response, *args, **kwargs)
        return self.response

    @property
    def paginator(self):
        """
        Small part of GenericAPIView
        """
        if not hasattr(self, "_paginator"):
            if self.pagination_class is None:
                self._paginator = None
            else:
                self._paginator = self.pagination_class()
                if self.page_size:
                    self._paginator.default_limit = self.page_size
                    self._paginator.page_size = self.page_size
        return self._paginator

    def finalize_response(self, request, response, *args, **kwargs):
        """
        Returns the final response object.
        """
        response = super().finalize_response(request, response, *args, **kwargs)
        exc = getattr(request, "exc", None)
        log_api_response(request, response, exc)
        return response
