from fan_ui.api.exceptions import InvalidEmailsError, ValidationError
from .common import *
from django.http import JsonResponse
from fan_ui.api.query_params import pass_account_object, pass_campaign_object
from fan.lists.csv_maillist import (
    parse_csv_data,
    EmailColumnNotFound,
    InvalidEmails,
    ReaderError,
    RecipientsCountExceeded,
    UserTemplateVariableLengthExceeded,
    UserTemplateVariableValueLengthExceeded,
    UserTemplateVariablesCountExceeded,
)
from fan.campaigns.get import get_csv_data


class CampaignRecipientListEndpoint(Endpoint):
    permission_classes = ()

    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    def get(self, request, account, campaign):
        csv_data = get_csv_data(campaign)
        recipients = []
        if csv_data:
            recipients = self._csv_to_recipient_list(csv_data)
        return JsonResponse(recipients, safe=False)

    def _csv_to_recipient_list(self, csv_data):
        try:
            maillist = parse_csv_data(csv_data)
        except InvalidEmails as exc:
            raise InvalidEmailsError(exc.invalid_emails)
        except EmailColumnNotFound:
            raise ValidationError({"data": "email_column_not_found"})
        except ReaderError:
            raise ValidationError({"data": "empty"})
        except RecipientsCountExceeded:
            raise ValidationError({"data": "too_long"})
        except UserTemplateVariablesCountExceeded:
            raise ValidationError({"data": "user_template_variables_count_exceeded"})
        except UserTemplateVariableLengthExceeded:
            raise ValidationError({"data": "user_template_variable_length_exceeded"})
        except UserTemplateVariableValueLengthExceeded:
            raise ValidationError({"data": "user_template_variable_value_length_exceeded"})

        recipients = []
        for row in maillist:
            row["email"] = row["email"].strip()
            recipient = {
                "email": row["email"],
                "user_template_params": self._row_to_user_template_params(row),
            }
            recipients.append(recipient)
        return recipients

    def _row_to_user_template_params(self, row):
        user_template_params = {}
        for key, value in list(row.items()):
            param = "user_variable_%s" % key
            user_template_params[param] = value
        return user_template_params
