from .common import *
from fan.campaigns.exceptions import (
    ForbiddenTargetCampaignState,
    ForbiddenCurrentCampaignState,
    UnsupportedStat,
    InvalidStatValue,
    MismatchedStatsSummary,
)
from fan.campaigns.set import set_campaign_send_result
from fan.models import Campaign
from fan.utils.persistent_log import persistent_log
from fan_ui.api.exceptions import ValidationError, WrongStateError
from fan_ui.api.query_params import (
    pass_account_object,
    pass_campaign_object,
    pass_campaign_state_param,
)


class CampaignStateEndpoint(Endpoint):
    permission_classes = ()

    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    @method_decorator(pass_campaign_state_param)
    def post(self, request, account, campaign, state):
        result = request.data
        self._validate_result(state, result)

        try:
            set_campaign_send_result(campaign, state, result)
        except ForbiddenTargetCampaignState:
            raise ValidationError({"state": "not_supported"})
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        except UnsupportedStat as exc:
            raise ValidationError({exc.stat: "not_supported"})
        except InvalidStatValue as exc:
            raise ValidationError({exc.stat: "invalid_type"})
        except MismatchedStatsSummary:
            raise ValidationError({"non_field": "mismatched_summary"})

        persistent_log(
            object_type="campaign",
            object_id=campaign.id,
            component="state",
            action="set_%s" % state,
            description="Изменен статус кампании",
            request=request,
            metadata={"error": result.get("error", "")},
        )

        return Response("OK", status=status.HTTP_200_OK)

    def _validate_result(self, state, result):
        if not isinstance(result, dict):
            raise ValidationError({"non_field": "invalid_type"})
        if state == Campaign.STATUS_FAILED:
            self._validate_failed_result(result)

    def _validate_failed_result(self, result):
        if "error" not in result:
            raise ValidationError({"error": "not_found"})
        if not result["error"]:
            raise ValidationError({"error": "empty"})
