from .common import *
from django.http import JsonResponse
from fan.accounts.get import get_account_from_logins
from fan.accounts.set import (
    set_account_from_logins,
    AccountFromLoginsEmpty,
    AccountFromLoginsTooLong,
    AccountFromLoginInvalid,
)
from fan.utils.permissions import additional_method_permissions
from fan_ui.api.exceptions import ValidationError
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
)
from fan_ui.api.endpoints_v1.permissions import AccountAdminPermission


class AccountFromLoginsEndpoint(Endpoint):
    permission_classes = (IsAuthenticated, ApiV1TvmServicePermission)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @additional_method_permissions((AccountAdminPermission,))
    def put(self, request, user_id, account):
        from_logins = request.data
        self._validate_list_of_strings(from_logins, "from_logins")
        try:
            set_account_from_logins(account, from_logins=from_logins)
        except AccountFromLoginsEmpty:
            raise ValidationError({"from_logins": "empty"})
        except AccountFromLoginsTooLong:
            raise ValidationError({"from_logins": "too_long"})
        except AccountFromLoginInvalid:
            raise ValidationError({"from_logins": "invalid_email_localpart"})
        return JsonResponse(account.from_logins, safe=False)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @additional_method_permissions((UserPermission,))
    def get(self, request, user_id, account):
        return JsonResponse(get_account_from_logins(account), safe=False)

    def _validate_list_of_strings(self, lst, name):
        if not isinstance(lst, list):
            raise ValidationError({name: "invalid_type"})
        for item in lst:
            if not isinstance(item, str):
                raise ValidationError({name: "invalid_type"})
