from .common import *
from django.http import JsonResponse
from fan.accounts.organizations.users import get_members_among
from fan.accounts.get import get_account_users
from fan.accounts.set import grant_account_access, revoke_account_access, AccountUsersLimitExceeded
from fan_ui.api.query_params import pass_user_id_param, pass_account_object
from fan_ui.api.exceptions import ValidationError, NotBelongsToOrg, TooManyElements
from fan_ui.api.endpoints_v1.permissions import AccountAdminPermission


class AccountUserListEndpoint(Endpoint):
    permission_classes = (IsAuthenticated, ApiV1TvmServicePermission, AccountAdminPermission)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    def get(self, request, user_id, account):
        users = get_account_users(account)
        return JsonResponse(users, safe=False)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    def post(self, request, user_id, account):
        user_ids_to_grant = request.data
        self._validate_list_of_strings(user_ids_to_grant, "uids")
        self._check_users_belongs_to_org(user_ids_to_grant, account.org_id)
        try:
            grant_account_access(account, user_ids_to_grant)
        except AccountUsersLimitExceeded:
            raise TooManyElements(detail="Account users limit exceeded")
        return Response("OK", status=status.HTTP_200_OK)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    def delete(self, request, user_id, account):
        user_ids_to_revoke = request.data
        self._validate_list_of_strings(user_ids_to_revoke, "uids")
        revoke_account_access(account, user_ids_to_revoke)
        return Response("OK", status=status.HTTP_200_OK)

    def _validate_list_of_strings(self, lst, name):
        if not isinstance(lst, list):
            raise ValidationError({name: "invalid_type"})
        for item in lst:
            if not isinstance(item, str):
                raise ValidationError({name: "invalid_type"})

    def _check_users_belongs_to_org(self, user_ids, org_id):
        org_users = set(get_members_among(org_id, user_ids))
        for user_id in user_ids:
            if user_id not in org_users:
                raise NotBelongsToOrg(
                    "user %s does not belong to organization %s" % (user_id, org_id)
                )
