from .common import *
from fan.models import Campaign
from fan.accounts.organizations.limits import is_draft_campaigns_limit_reached
from fan.campaigns.create import create_campaign, clone_campaign
from fan.campaigns.delete import delete_campaign
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan_ui.api.exceptions import WrongStateError, DraftCampaignsLimitReached
from fan_ui.api.serializers.campaign import CampaignDetailedSerializerV1
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
    pass_campaign_object,
    pass_source_campaign_object,
)


class CampaignEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    def get(self, request, user_id, account, campaign):
        return make_ok_response(campaign, CampaignDetailedSerializerV1)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_source_campaign_object(required=False))
    def post(self, request, user_id, account, source_campaign):
        self._check_draft_campaigns_limit_reached(account)
        if source_campaign:
            campaign = clone_campaign(user_id, account, source_campaign)
        else:
            campaign = create_campaign(
                account=account,
                owner=user_id,
                type=Campaign.TYPE_SIMPLE,
                state=Campaign.STATUS_DRAFT,
            )
        return make_ok_response(campaign, CampaignDetailedSerializerV1)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    def delete(self, request, user_id, account, campaign):
        try:
            delete_campaign(campaign)
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        return Response("OK", status=status.HTTP_200_OK)

    def _check_draft_campaigns_limit_reached(self, account):
        # This doesn't guarantee strict compliance with the limit in case of parallel requests.
        # In the worst case, there may be (LIMIT + N_INSTANCES * N_WORKERS - 1) drafts
        if is_draft_campaigns_limit_reached(account.org_id):
            raise DraftCampaignsLimitReached()
