from .common import *
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan.campaigns.set import set_campaign_details
from fan.utils.emails import is_valid
from fan_ui.api.query_params import pass_account_object, pass_campaign_object, pass_user_id_param
from fan_ui.api.exceptions import ValidationError, WrongStateError
from fan_ui.api.serializers.campaign import CampaignDetailedSerializerV1


class CampaignDetailsEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)

    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    @method_decorator(pass_user_id_param)
    def put(self, request, account, campaign, user_id):
        campaign_details = request.data
        self._validate_campaign_details(campaign_details)
        try:
            set_campaign_details(campaign, campaign_details)
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        return make_ok_response(campaign, CampaignDetailedSerializerV1)

    def _validate_campaign_details(self, campaign_details):
        self._validate_is_object(campaign_details)
        self._validate_is_not_empty(campaign_details)
        self._validate_campaign_details_are_supported(campaign_details)
        self._validate_from_email(campaign_details)
        self._validate_from_name(campaign_details)
        self._validate_subject(campaign_details)

    def _validate_is_object(self, campaign_details):
        if not isinstance(campaign_details, dict):
            raise ValidationError({"non_field": "invalid_type"})

    def _validate_is_not_empty(self, campaign_details):
        if len(campaign_details) == 0:
            raise ValidationError({"non_field": "empty_params"})

    def _validate_campaign_details_are_supported(self, campaign_details):
        for field in campaign_details:
            if field not in (
                "from_email",
                "from_name",
                "subject",
            ):
                raise ValidationError({field: "not_supported"})

    def _validate_from_email(self, campaign_details):
        from_email = campaign_details.get("from_email")
        if from_email is None:
            return
        self._check_is_string("from_email", from_email)
        self._check_max_length("from_email", from_email, 384)
        if not is_valid(from_email):
            raise ValidationError({"from_email": "invalid_email"})

    def _validate_from_name(self, campaign_details):
        from_name = campaign_details.get("from_name")
        if from_name is None:
            return
        self._check_is_string("from_name", from_name)
        self._check_max_length("from_name", from_name, 384)

    def _validate_subject(self, campaign_details):
        subject = campaign_details.get("subject")
        if subject is None:
            return
        self._check_is_string("subject", subject)
        self._check_max_length("subject", subject, 1024)

    def _check_is_string(self, field, value):
        if not isinstance(value, str) and not isinstance(value, str):
            raise ValidationError({field: "invalid_type"})

    def _check_max_length(self, field, value, max_length):
        if len(value) > max_length:
            raise ValidationError({field: "too_long"})
