from zipfile import BadZipfile
from django.http import HttpResponse

from rest_framework.parsers import MultiPartParser

from fan.links.unsubscribe import UNSUBSCRIBE_LINK
from fan.message.letter import (
    load_letter,
    check_letter_html_data_len,
    check_letter_zip_data_len,
    SUPPORTED_CONTENT_TYPES,
    CONTENT_TYPE_HTML,
    CONTENT_TYPE_ZIP,
    AttachmentsCountExceeded,
    AttachmentsSizeExceeded,
    UserTemplateVariablesCountExceeded,
    UserTemplateVariableLengthExceeded,
    CONTENT_TYPE_MULTIPART,
)
from fan.message.exceptions import TemplateRuntimeError, UnsubscribeLinkNotFoundError
from fan.message.loader import HTMLWrongFormat
from fan.models import Campaign
from fan.utils.zip import create_zip_file
from .common import *
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
    pass_campaign_object,
)
from fan_ui.api.exceptions import (
    ValidationError,
    WrongStateError,
    RenderApiError,
    ResourceDoesNotExist,
)
from fan_ui.api.parsers import FileUploadParser
from fan_ui.api.serializers.letter import LetterSummarySerializerV1


class CampaignLetterEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)
    parser_classes = (MultiPartParser, FileUploadParser)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    def put(self, request, user_id, account, campaign):
        """
        Загружает тело письма в заранее созданный campaign.default_letter.
        Если письмо перадаётся с аттачами, то аттачи загружаются в базу и публикуются.

        ATTENTION: В коде этой ручки проверяется ограничение на размер письма, однако
            в DRF нет стриминга. Так что, такая проверка уже поздновата - файл загружен
            в память. Поэтому подразумевается, что некоторый
            внешний слой должен заботиться о лимитах (например nginx).
        """
        uploaded_file = self._get_uploaded_file(request)
        self._validate_campaign_state(campaign)
        letter = campaign.default_letter
        self._load_letter(letter, uploaded_file)
        return make_ok_response(letter, LetterSummarySerializerV1)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    def get(self, request, user_id, account, campaign):
        letter = campaign.default_letter
        if len(letter.html_body) == 0:
            raise ResourceDoesNotExist({"letter": "empty"})
        return HttpResponse(letter.html_body)

    def _get_uploaded_file(self, request):
        if CONTENT_TYPE_MULTIPART in request.content_type:
            if len(request.data) == 0:
                raise ValidationError({"data": "empty"})
            uploaded_file = self._make_uploaded_file_from_multipart(request)
        else:
            if "file" not in request.data:
                raise ValidationError({"data": "empty"})
            uploaded_file = request.data["file"]
        if uploaded_file.content_type not in SUPPORTED_CONTENT_TYPES:
            raise ValidationError({"content_type": "not_supported"})
        self._check_data_len(uploaded_file)
        return uploaded_file

    def _make_uploaded_file_from_multipart(self, request):
        files = self._get_files_from_request(request)
        uploaded_file = create_zip_file(files, "letter.zip")
        return uploaded_file

    def _get_files_from_request(self, request):
        files = {}
        for name in request.FILES:
            for file in request.FILES.getlist(name):
                files[file.name] = file
        return files

    def _validate_campaign_state(self, campaign):
        if campaign.state != Campaign.STATUS_DRAFT:
            raise WrongStateError(campaign.state, Campaign.STATUS_DRAFT)

    def _load_letter(self, letter, uploaded_file):
        try:
            load_letter(letter=letter, html_body_file=uploaded_file)
        except HTMLWrongFormat:
            raise ValidationError({"data": "invalid_email"})
        except BadZipfile:
            raise ValidationError({"data": "invalid_zip"})
        except UnsubscribeLinkNotFoundError:
            raise ValidationError({UNSUBSCRIBE_LINK: "not_found"})
        except AttachmentsCountExceeded:
            raise ValidationError({"data": "too_long"})
        except AttachmentsSizeExceeded:
            raise ValidationError({"data": "too_long"})
        except UserTemplateVariablesCountExceeded:
            raise ValidationError({"data": "user_template_variables_count_exceeded"})
        except UserTemplateVariableLengthExceeded:
            raise ValidationError({"data": "user_template_variable_length_exceeded"})
        except TemplateRuntimeError as e:
            raise RenderApiError(code="template_runtime_error", description=str(e))

    def _check_data_len(self, letter_file):
        content_type = letter_file.content_type
        data_len = len(letter_file)
        if (content_type == CONTENT_TYPE_HTML and not check_letter_html_data_len(data_len)) or (
            content_type == CONTENT_TYPE_ZIP and not check_letter_zip_data_len(data_len)
        ):
            raise ValidationError({"data": "too_long"})
