from .common import *
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
    pass_campaign_object,
    pass_maillist_object,
)
from fan.utils.csv import UnsupportedFileFormat, csv_data_from_file
from fan_ui.api.exceptions import ValidationError, WrongStateError, InvalidEmailsError
from fan_ui.api.parsers import FileUploadParser
from fan_ui.api.serializers.campaign import CampaignDetailedSerializerV1
from fan_ui.api.serializers.maillist import SingleUseMailListSummarySerializerV1
from fan.lists.csv_maillist import (
    EmailColumnNotFound,
    ReaderError,
    RecipientsCountExceeded,
    UserTemplateVariablesCountExceeded,
    UserTemplateVariableLengthExceeded,
    UserTemplateVariableValueLengthExceeded,
    InvalidEmails,
)
from fan.lists.maillist import set_maillist_for_campaign
from fan.lists.singleuse import store_csv_maillist_for_campaign


class CampaignMailListEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)
    parser_classes = (FileUploadParser,)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    @method_decorator(pass_maillist_object(required=False))
    def put(self, request, user_id, account, campaign, maillist):
        if maillist:
            try:
                campaign = set_maillist_for_campaign(campaign, maillist)
            except ForbiddenCurrentCampaignState as exc:
                raise WrongStateError(exc.actual_state, exc.required_state)
            return make_ok_response(campaign, CampaignDetailedSerializerV1)

        uploaded_file = self._get_uploaded_file(request)
        csv_data = self._get_csv_data(uploaded_file)

        try:
            store_csv_maillist_for_campaign(campaign, csv_data, uploaded_file.name)
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        except InvalidEmails as exc:
            raise InvalidEmailsError(exc.invalid_emails)
        except EmailColumnNotFound:
            raise ValidationError({"data": "email_column_not_found"})
        except ReaderError:
            raise ValidationError({"data": "empty"})
        except RecipientsCountExceeded:
            raise ValidationError({"data": "too_long"})
        except UserTemplateVariablesCountExceeded:
            raise ValidationError({"data": "user_template_variables_count_exceeded"})
        except UserTemplateVariableLengthExceeded:
            raise ValidationError({"data": "user_template_variable_length_exceeded"})
        except UserTemplateVariableValueLengthExceeded:
            raise ValidationError({"data": "user_template_variable_value_length_exceeded"})

        return make_ok_response(campaign.single_use_maillist, SingleUseMailListSummarySerializerV1)

    def _get_uploaded_file(self, request):
        if "file" not in request.data:
            raise ValidationError({"data": "not_found"})
        return request.data["file"]

    def _get_csv_data(self, uploaded_file):
        try:
            return csv_data_from_file(uploaded_file)
        except UnsupportedFileFormat:
            raise ValidationError({"data": "not_supported"})
