from .common import *
from fan.accounts.organizations.limits import check_campaign_fits_to_send_limits, CheckLimitsResult
from fan.campaigns.exceptions import (
    ForbiddenTargetCampaignState,
    ForbiddenCurrentCampaignState,
    NoMaillist,
    NoLetter,
    EmptyFromEmail,
    EmptyFromName,
    EmptySubject,
    DomainNotBelongs,
    DomainNoMX,
    DomainNoDKIM,
    DomainNoSPF,
    LoginNotBelongs,
)
from fan.campaigns.set import set_campaign_state
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
    pass_campaign_object,
    pass_campaign_state_param,
)
from fan_ui.api.exceptions import (
    ValidationError,
    WrongStateError,
    NotReadyError,
    WrongDomainError,
    SendLimitReached,
    WrongLoginError,
)
from fan_ui.typed_log import update_custom_log_fields


class CampaignStateEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object)
    @method_decorator(pass_campaign_state_param)
    def post(self, request, user_id, account, campaign, state):
        check_result, diagnostic_info = check_campaign_fits_to_send_limits(campaign)
        if check_result == CheckLimitsResult.CAMPAIGNS_PER_DAY_LIMIT_REACHED:
            update_custom_log_fields(request, diagnostic_info)
            raise SendLimitReached()  # TODO change detail to 'campaigns_per_day_limit'
        elif check_result == CheckLimitsResult.EMAILS_PER_MONTH_LIMIT_REACHED:
            update_custom_log_fields(request, diagnostic_info)
            raise SendLimitReached()  # TODO change detail to 'emails_per_month_limit'

        try:
            set_campaign_state(campaign, state)
        except ForbiddenTargetCampaignState:
            raise ValidationError({"state": "not_supported"})
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        except NoMaillist:
            raise NotReadyError("no_maillist")
        except NoLetter:
            raise NotReadyError("no_letter")
        except EmptyFromEmail:
            raise NotReadyError("empty_from_email")
        except EmptyFromName:
            raise NotReadyError("empty_from_name")
        except EmptySubject:
            raise NotReadyError("empty_subject")
        except DomainNotBelongs:
            raise WrongDomainError("not_belongs")
        except DomainNoMX:
            raise WrongDomainError("no_mx")
        except DomainNoDKIM:
            raise WrongDomainError("no_dkim")
        except DomainNoSPF:
            raise WrongDomainError("no_spf")
        except LoginNotBelongs:
            raise WrongLoginError("not_belongs")
        return Response("OK", status=status.HTTP_200_OK)
