from .common import *
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan_ui.api.query_params import (
    pass_user_id_param,
    pass_account_object,
    pass_campaign_object,
    pass_maillist_object,
    pass_title_param,
)
from fan.utils.csv import UnsupportedFileFormat, csv_data_from_file
from fan_ui.api.exceptions import (
    ValidationError,
    WrongStateError,
    InvalidEmailsError,
    MaillistsLimitReached,
)
from fan_ui.api.parsers import FileUploadParser
from fan_ui.api.serializers.maillist import DetailedMaillistSerializer
from fan.lists.csv_maillist import (
    EmailColumnNotFound,
    ReaderError,
    RecipientsCountExceeded,
    UserTemplateVariablesCountExceeded,
    UserTemplateVariableLengthExceeded,
    UserTemplateVariableValueLengthExceeded,
    InvalidEmails,
)
from fan.lists.maillist import (
    set_maillist_details,
    store_maillist,
    delete_maillist,
    MaillistTitleLengthExceeded,
    MaillistTitleDuplicated,
    MaillistsCountExceeded,
)


class MaillistEndpoint(Endpoint):
    permission_classes = (UserPermission, IsAuthenticated, ApiV1TvmServicePermission)
    parser_classes = (FileUploadParser,)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_maillist_object)
    def get(self, request, user_id, account, maillist):
        return make_ok_response(maillist, DetailedMaillistSerializer)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_maillist_object)
    @method_decorator(pass_title_param)
    def put(self, request, user_id, account, maillist, title):
        try:
            maillist = set_maillist_details(maillist, title)
        except MaillistTitleLengthExceeded:
            raise ValidationError({"title": "too_long"})
        except MaillistTitleDuplicated:
            raise ValidationError({"title": "duplicated"})
        return make_ok_response(maillist, DetailedMaillistSerializer)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_campaign_object(required=False))
    @method_decorator(pass_title_param(required=False, blank=True))
    def post(self, request, user_id, account, campaign, title):
        uploaded_file = self._get_uploaded_file(request)
        csv_data = self._get_csv_data(uploaded_file)
        try:
            maillist = store_maillist(
                account, csv_data, uploaded_file.name, campaign=campaign, title=title
            )
        except InvalidEmails as exc:
            raise InvalidEmailsError(exc.invalid_emails)
        except EmailColumnNotFound:
            raise ValidationError({"data": "email_column_not_found"})
        except ReaderError:
            raise ValidationError({"data": "empty"})
        except RecipientsCountExceeded:
            raise ValidationError({"data": "too_long"})
        except UserTemplateVariablesCountExceeded:
            raise ValidationError({"data": "user_template_variables_count_exceeded"})
        except UserTemplateVariableLengthExceeded:
            raise ValidationError({"data": "user_template_variable_length_exceeded"})
        except UserTemplateVariableValueLengthExceeded:
            raise ValidationError({"data": "user_template_variable_value_length_exceeded"})
        except ForbiddenCurrentCampaignState as exc:
            raise WrongStateError(exc.actual_state, exc.required_state)
        except MaillistTitleLengthExceeded:
            raise ValidationError({"title": "too_long"})
        except MaillistTitleDuplicated:
            raise ValidationError({"title": "duplicated"})
        except MaillistsCountExceeded:
            raise MaillistsLimitReached()
        return make_ok_response(maillist, DetailedMaillistSerializer)

    @method_decorator(pass_user_id_param)
    @method_decorator(pass_account_object)
    @method_decorator(pass_maillist_object)
    def delete(self, request, user_id, account, maillist):
        delete_maillist(maillist)
        return Response("OK", status=status.HTTP_200_OK)

    def _get_uploaded_file(self, request):
        if "file" not in request.data:
            raise ValidationError({"data": "not_found"})
        return request.data["file"]

    def _get_csv_data(self, uploaded_file):
        try:
            return csv_data_from_file(uploaded_file)
        except UnsupportedFileFormat:
            raise ValidationError({"data": "not_supported"})
