import logging
import sys
from rest_framework import status
from rest_framework.exceptions import APIException, NotAuthenticated, PermissionDenied
from rest_framework.response import Response
from rest_framework.serializers import ValidationError
from rest_framework.views import exception_handler

from fan.exceptions import PermanentError


class BadRequestError(APIException):
    status_code = status.HTTP_400_BAD_REQUEST

    def __init__(self, error, detail):
        self.error = error
        self.detail = detail


class InvalidEmailsError(BadRequestError):
    def __init__(self, envalid_emails):
        super().__init__("invalid_emails", envalid_emails)


class ConflictError(APIException):
    status_code = status.HTTP_409_CONFLICT


class LimitReached(APIException):
    status_code = status.HTTP_429_TOO_MANY_REQUESTS
    error = "limit_reached"


class SendLimitReached(LimitReached):
    default_detail = "send_limit_reached"


class TestSendLimitReached(LimitReached):
    default_detail = "test_send_limit_reached"


class DraftCampaignsLimitReached(LimitReached):
    default_detail = "draft_campaigns_limit_reached"


class MaillistsLimitReached(LimitReached):
    default_detail = "maillists_limit_reached"


class TooManyElements(ConflictError):
    error = "too_many_elements"


class WrongStateError(ConflictError):
    error = "wrong_state"

    def __init__(self, actual_state, required_state):
        super().__init__({"actual_state": actual_state, "required_state": required_state})


class WrongDomainError(ConflictError):
    error = "wrong_domain"


class WrongLoginError(ConflictError):
    error = "wrong_login"


class NotBelongsToOrg(ConflictError):
    error = "not_belongs_to_org"


class NotReadyError(ConflictError):
    error = "not_ready"


class ResourceDoesNotExist(APIException):
    status_code = 404
    error = "resource_does_not_exist"

    def __init__(self, detail=None, code=None):
        self.detail = detail
        self.code = code


class RenderApiError(APIException):
    status_code = 400
    error = "render"
    default_detail = "Render error"

    def __init__(self, code=None, description=None):
        self.detail = description
        self.code = code
        self.description = description


def api_exception_handler(exc, context):
    _log_exception(context)
    response = exception_handler(exc, context)
    if response is None:
        response = Response(data={}, status=_status_by_exception(exc))

    # This hack is used in Endpoint.finalize_response for logging
    exc.traceback = sys.exc_info()[2]
    request = context["request"]
    request.exc = exc

    if isinstance(exc, BadRequestError):
        response.data = {"error": exc.error, "detail": exc.detail}

    if isinstance(exc, ValidationError):
        response.data = {"error": "validation", "detail": response.data}

    if isinstance(exc, ConflictError):
        detail = response.data["detail"] if "detail" in response.data else response.data
        response.data = {"error": exc.error, "detail": detail}

    if isinstance(exc, ResourceDoesNotExist):
        response.data = {"error": exc.error, "detail": response.data}

    if isinstance(exc, RenderApiError):
        response.data = {
            "error": exc.error,
            "detail": {
                "code": exc.code,
                "description": exc.description,
            },
        }

    if isinstance(exc, NotAuthenticated):
        response.data = {
            "error": "not_authenticated",
            "detail": exc.detail,
        }

    if isinstance(exc, PermissionDenied):
        response.data = {"error": "forbidden", "detail": response.data["detail"]}

    if isinstance(exc, LimitReached):
        response.data = {"error": exc.error, "detail": response.data["detail"]}

    if "error" not in response.data:
        _add_error_code(response)

    return response


def _status_by_exception(exc):
    if isinstance(exc, PermanentError):
        return 400
    else:
        return 500


def _add_error_code(response):
    if response.status_code // 100 == 4:
        response.data["error"] = "bad_request"
    elif response.status_code // 100 == 5:
        response.data["error"] = "internal_server_error"


def _log_exception(context):
    request = context["request"]
    logging.exception("API error for request %s %s", request.method, request.path)
