from functools import wraps

from fan.accounts.get import get_account_by_name
from fan.campaigns.get import get_campaign_by_slug
from fan.lists.maillist import get_maillist, MaillistDoesNotExist
from fan.send.test_send import get_by_id as get_test_send_task_by_id
from fan.models import TestSendTask as TSendTask
from fan.models import Account, Campaign
from fan_ui.api.exceptions import ValidationError, ResourceDoesNotExist

# pass_X_param decorators
# do query param X lookup, validation and then pass its value to wrapped function args list.

# pass_X_object decorators
# lookup an object of model class X and pass it to wrapped function args list.


def pass_user_id_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            user_id = _get_param(request, "user_id", required, blank)
            kwargs["user_id"] = user_id
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_account_slug_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            account_slug = _get_param(request, "account_slug", required, blank)
            kwargs["account_slug"] = account_slug
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_campaign_slug_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            campaign_slug = _get_param(request, "campaign_slug", required, blank)
            kwargs["campaign_slug"] = campaign_slug
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_campaign_state_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            state = _get_param(request, "state", required, blank)
            _validate_campaign_state(state)
            kwargs["state"] = state
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_states_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            states_str = _get_param(request, "states", required, blank)
            states = _parse_states_param(states_str)
            kwargs["states"] = states
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_org_id_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            org_id = _get_param(request, "org_id", required, blank)
            kwargs["org_id"] = org_id
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_recipient_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            kwargs["recipient"] = _get_param(request, "recipient", required, blank)
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_count_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            count = _get_param(request, "count", required, blank)
            if count:
                if not count.lstrip("-").isdigit():
                    raise ValidationError({"count": "invalid_type"})
                count = int(count)
                if count <= 0:
                    raise ValidationError({"count": "invalid_value"})
            kwargs["count"] = count
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_title_param(func=None, required=True, blank=False):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            title = _get_param(request, "title", required, blank)
            kwargs["title"] = title
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_account_object(func=None, required=True):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            account_slug = _get_param(request, "account_slug", required)
            account = _get_account(account_slug, required)
            kwargs["account"] = account
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_campaign_object(
    func=None, required=True, url_param="campaign_slug", handler_arg="campaign"
):
    """
    This decorator requires 'account' argument in a wrapped function.
    Hence it should be used in pair with pass_account_object decorator.
    """

    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            campaign_slug = _get_param(request, url_param, required)
            if "account" not in kwargs:
                raise TypeError(detail="account param not parsed")
            account = kwargs["account"]
            campaign = _get_campaign(campaign_slug, account.id, required)
            kwargs[handler_arg] = campaign
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_source_campaign_object(func=None, required=True):
    return pass_campaign_object(
        func,
        required,
        url_param="source_campaign_slug",
        handler_arg="source_campaign",
    )


def pass_maillist_object(
    func=None, required=True, url_param="maillist_slug", handler_arg="maillist"
):
    """
    This decorator requires 'account' argument in a wrapped function.
    Hence it should be used in pair with pass_account_object decorator.
    """

    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            slug = _get_param(request, url_param, required)
            if "account" not in kwargs:
                raise TypeError(detail="account param not parsed")
            account = kwargs["account"]
            maillist = _get_maillist(account, slug, required)
            kwargs[handler_arg] = maillist
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_test_send_task_object(func=None, required=True):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            task_id = _get_param(request, "task_id", required)
            task = _get_test_send_task(task_id, required)
            kwargs["task"] = task
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_limit_value_param(func=None, required=True):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            value = _get_param(request, "value", required)
            value = _parse_limit_value_param(value)
            _validate_limit_value_param(value)
            kwargs["value"] = value
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def pass_trusty_value_param(func=None, required=True):
    def decorator(f):
        @wraps(f)
        def wrapper(request, *args, **kwargs):
            value = _get_param(request, "value", required)
            value = _parse_trusty_value_param(value)
            kwargs["value"] = value
            return f(request, *args, **kwargs)

        return wrapper

    return decorator if func is None else decorator(func)


def _get_param(request, param_name, required, blank=False):
    result = request.query_params.get(param_name, None)
    if required and result is None:
        raise ValidationError({param_name: "not_found"})
    if not blank and result is not None and len(result) == 0:
        raise ValidationError({param_name: "empty"})
    return result


def _parse_states_param(states_str):
    if states_str is None:
        return None
    states = [_f for _f in states_str.split(",") if _f]  # removes empty/None elements
    for state in states:
        _validate_campaign_state(state)
    return tuple(states)


def _validate_campaign_state(state):
    if state not in Campaign.VALID_GLOBAL_STATES:
        raise ValidationError({"state": "not_supported"})


def _get_account(account_slug, required):
    try:
        return get_account_by_name(account_slug)
    except Account.DoesNotExist:
        if required or account_slug:
            raise ResourceDoesNotExist(detail="account not found")
        return None


def _get_campaign(campaign_slug, account_id, required):
    try:
        return get_campaign_by_slug(campaign_slug, account_id)
    except Campaign.DoesNotExist:
        if required or campaign_slug:
            raise ResourceDoesNotExist(detail="campaign not found")
        return None


def _get_maillist(account, slug, required):
    try:
        return get_maillist(account, slug)
    except MaillistDoesNotExist:
        if required or slug:
            raise ResourceDoesNotExist(detail="maillist not found")
        return None


def _get_test_send_task(task_id, required):
    try:
        return get_test_send_task_by_id(task_id)
    except TSendTask.DoesNotExist:
        if required or task_id:
            raise ResourceDoesNotExist(detail="test send task not found")
        return None


def _parse_limit_value_param(value):
    try:
        return int(value)
    except:
        raise ValidationError({"value": "invalid_type"})


def _validate_limit_value_param(value):
    if value < 0:
        raise ValidationError({"value": "invalid_value"})


def _parse_trusty_value_param(value):
    if value == "true":
        return True
    if value == "false":
        return False
    raise ValidationError({"value": "invalid_type"})
